unit Main;

interface

uses Classes, blcksock, TlntSend, PINGsend, SysUtils, IniFiles, Libc;

const
  cDefNumTimeouts = 5;
  cDefTimeout = 1000;
  cNumHosts = 4;
  cDefTimer = 2000;
  cIniFileName = 'watchdog.ini';
  cLogFileName = 'watchdog.log';
  cKomFileName = 'comm.log';
  cResetTimeout = 60000; // minuta

type
  TThReset = class(TThread)
  private
    Telnet: TTelnetSend;
    Start: TDateTime;
  public
    Krok: integer;
    constructor Create;
    procedure Execute; override;
    destructor Destroy; override;
  end;

  TPingTh = class(TThread)
  private
    PingSend: TPINGSend;
    ID: Integer; // ID Threadu
    function GetTimeout: Integer;
    procedure SetTimeout(value: Integer);
  public
    Host: string;
    property Timeout: Integer read GetTimeout write SetTimeout;

    constructor Create(const Host: string; const Timeout: Cardinal; fID: integer);
    procedure Execute; override;
    destructor Destroy; override;
  end;

  TPingObj = class(TObject)
  private
    Count, PLCount: Cardinal;
    Avg: Real;
    ID: integer;
    PLInRow: Integer;
  public
    Addr: string;
    CanReset: Boolean;
    procedure AddPing(const time: Integer);
    constructor Create(fID: integer);
  end;

procedure Log(str: string);
procedure ResetError;
procedure ResetStart;
procedure ResetOK;
procedure ResetProblem;
procedure ProgramEnd;
procedure ProgramStart;

implementation

{ TPingTh }

var
  semReset, semLog: sem_t;
  ThReset: TThReset;
  cTimer: integer;
  cNumTimeouts: Integer;
  ResetHost: string;
  Challenges, Answers: TStrings;
  PingObjs: array[0..cNumHosts-1] of TPingObj;
  PingThs: array[0..cNumHosts-1] of TPingTh;

constructor TPingTh.Create(const Host: string; const Timeout: Cardinal; fID: integer);
begin
  inherited Create(True);

  FreeOnTerminate := False;
  ID := fID;
  self.Host := Host;
  PingSend := TPingSend.Create;
  PingSend.Timeout := TimeOut;
  self.Timeout := timeout;
end;

destructor TPingTh.Destroy;
begin
  PingSend.Free;

  inherited;
end;

procedure TPingTh.Execute;
var
  poc: integer;
begin
  poc := cTimer;
  while not Terminated do
  begin
    Dec(poc, 100); Sleep(100);
    if poc < 0 then
    begin
      poc := cTimer;
      try
        if PingSend.Ping(Host) then
          PingObjs[ID].AddPing(PingSend.PingTime)
        else
          PingObjs[ID].AddPing(-1);
      except
        PingObjs[ID].AddPing(-1);
      end;
    end;
  end;
end;

function TPingTh.GetTimeout: Integer;
begin
  Result := PingSend.Timeout;
end;

procedure TPingTh.SetTimeout(value: Integer);
begin
  PingSend.Timeout := value;
end;

{ TPingObj }

procedure TPingObj.AddPing(const time: Integer);
var
  i: integer;
  Reset: Boolean;
begin
  if Time = -1 then
  begin
    Inc(PLCount);
    Inc(PLInRow);

    if CanReset and (PLInRow >= cNumTimeouts) then
    begin
      Reset := True;
      for i := 0 to cNumHosts-1 do
      begin
        if PingObjs[i].CanReset and (PingObjs[i].PLInRow < cNumTimeOuts) then
        begin
          Reset := False;
          break;
        end;
      end;

      if Reset then
      begin
        if sem_trywait(semReset) = 0 then
        begin
          if ThReset <> nil then ThReset.Free;
          ThReset := TThReset.Create;
        end;

        for i := 0 to cNumHosts-1 do
          PingObjs[i].PLInRow := 0;
      end;
    end;
  end
  else
  begin
    Inc(Count);
    Avg := (Avg * (Count - 1) + Time) / Count;
    PLInRow := 0;
  end;

//  WriteLn(Addr + ' ' + IntToStr(time) + ' PL:' + IntToStr(PLInRow));
end;

constructor TPingObj.Create(fID: integer);
begin
  Avg := 0;
  Count := 0;
  PLCount := 0;
  PLInRow := 0;
  ID := fID;
  CanReset := True;
end;

function CurDir: string;
var
  Dir: string;
begin
  GetDir(0, Dir);
  if Dir[Length(Dir)] <> '/' then Dir := Dir + '/';
  Result := Dir;
end;

{ TThReset }

constructor TThReset.Create;
begin
  inherited Create(True);

  Telnet := TTelnetSend.Create;
  Telnet.TargetHost := ResetHost;
  Telnet.TargetPort := '23';
  Telnet.Timeout := 1000;
  Krok := 0;
  FreeOnTerminate := False;
  Resume;
end;

destructor TThReset.Destroy;
begin
  Telnet.Free;
end;

procedure TThReset.Execute;

  procedure Log(s: string);
  var
    LogFile: TFileStream;
    Mode: Cardinal;
  begin
    if FileExists(CurDir + cKomFileName) then
      Mode := fmOpenWrite
    else
      Mode := fmCreate;

    try
      LogFile := TFileStream.Create(CurDir + cKomFileName, Mode);
    except
      Exit;
    end;

    try
      s := s + #$0d#$0a;
      LogFile.Seek(0, soFromEnd);
      LogFile.Write(s[1], Length(s));
    finally
      LogFile.Free;
    end;
  end;

var
  str: string;
begin
  Start := Now;
  DeleteFile(CurDir + cKomFileName);
  ResetStart;
  try
    try
      if not Telnet.Login then
      begin
        ResetError;
        Exit;
      end;
    except
      begin
        ResetError;
        Exit;
      end;
    end;

    try
      while not Terminated and (Krok < Challenges.Count) and
        (Start + cResetTimeout/24/60/60/1000 > Now) do
      begin
        try
          str := Telnet.RecvTerminated(Challenges[Krok]);
        except
          break;
        end;

        if str <> '' then
        begin
          Log('Step #' + IntToStr(Krok+1));
          Log(str + Challenges[Krok]);
          Log('-> ' + Answers[Krok]);
          Telnet.Send(Answers[Krok] + #$0d#$0a);
          Inc(Krok);
        end;
      end;

      if Krok = Challenges.Count then
        ResetOK
      else
        ResetProblem;
    finally
      Telnet.Logout;
    end;
  finally
    sem_post(semReset);
  end;
end;

procedure ResetError;
var
  Text: string;
begin
  Text := 'Unable to reset, cannot connect to ' + ResetHost;
  Log(Text);
end;

procedure ResetOK;
var
  Text: string;
begin
  Text := 'Reset OK';
  Log(Text);
end;

procedure Log(str: string);
var
  LogFile: TFileStream;
  Mode: Cardinal;
begin
  sem_wait(semLog);
  try
//    WriteLn(str);

    if FileExists(CurDir + cLogFileName) then
      Mode := fmOpenWrite
    else
      Mode := fmCreate;

    try
      LogFile := TFileStream.Create(CurDir + cLogFileName, Mode);
    except
      Exit;
    end;

    try
      str := DateTimeToStr(Now) + ': ' + str + #$0d#$0a;
      LogFile.Seek(0, soFromEnd);
      LogFile.Write(str[1], Length(str));
    finally
      LogFile.Free;
    end;
  finally
    sem_post(semLog);
  end;
end;

procedure ResetStart;
begin
//  Writeln('Trying to reset...');
end;

procedure ResetProblem;
var
  Text: string;
begin
  Text := 'Unable to reset, invalid response in step #' + IntToStr(ThReset.Krok+1);
  Log(Text);
end;

procedure ProgramStart;
var
  i: integer;
  IniFile: TIniFile;
  hostname: string;
  timeout: integer;
  ChCnt: integer;
begin
  PingObjs[0] := TPingObj.Create(0);
  PingObjs[1] := TPingObj.Create(1);
  PingObjs[2] := TPingObj.Create(2);
  PingObjs[3] := TPingObj.Create(3);

  sem_init(semReset, False, 1);
  sem_init(semLog, False, 1);
  ThReset := nil;

  Log('Starting...');

  IniFile := TIniFile.Create(CurDir + cIniFileName);

  cNumTimeouts := IniFile.ReadInteger('Constants', 'MaxTimeouts', cDefNumTimeouts);
  IniFile.WriteInteger('Constants', 'MaxTimeouts', cNumTimeouts);

  cTimer := IniFile.ReadInteger('Constants', 'PingInterval', cDefTimer);
  IniFile.WriteInteger('Constants', 'PingInterval', cTimer);

  ResetHost := IniFile.ReadString('ResetHost', 'Hostname', '192.168.168.1');
  IniFile.WriteString('ResetHost', 'Hostname', ResetHost);

  Challenges := TStringList.Create;
  Answers := TStringList.Create;
  ChCnt := IniFile.ReadInteger('ResetHost', 'ChallengesCount', 0);
  if ChCnt = 0 then
  begin
    ChCnt := 3;
    IniFile.WriteInteger('ResetHost', 'ChallengesCount', 3);
    IniFile.WriteString('ResetHost', 'Challenge1', 'password:');
    IniFile.WriteString('ResetHost', 'Answer1', 'password');
    IniFile.WriteString('ResetHost', 'Challenge2', 'Command>');
    IniFile.WriteString('ResetHost', 'Answer2', 'reset system');
    IniFile.WriteString('ResetHost', 'Challenge3', '(y/n)?');
    IniFile.WriteString('ResetHost', 'Answer3', 'y');
  end;

  for i := 1 to ChCnt do
  begin
    Challenges.Add(IniFile.ReadString('ResetHost', 'Challenge'+IntToStr(i),''));
    Answers.Add(IniFile.ReadString('ResetHost', 'Answer'+IntToStr(i),''));
  end;

  PingObjs[0].CanReset := IniFile.ReadBool('Host1', 'TimeoutDoReset', True);
  PingObjs[1].CanReset := IniFile.ReadBool('Host2', 'TimeoutDoReset', True);
  PingObjs[2].CanReset := IniFile.ReadBool('Host3', 'TimeoutDoReset', True);
  PingObjs[3].CanReset := IniFile.ReadBool('Host4', 'TimeoutDoReset', False);

  for i := 0 to cNumHosts - 1 do
  begin
    hostname := IniFile.ReadString('Host' + IntToStr(i+1), 'Hostname', '');
    timeout := IniFile.ReadInteger('Host' + IntToStr(i+1), 'Timeout', cDefTimeout);

    if Hostname = '' then
    begin
      case i+1 of
        1: Hostname := 'ntp.karpo.cz';
        2: Hostname := 'www.seznam.cz';
        3: Hostname := 'www.cesnet.cz';
        4: Hostname := '192.168.168.1';
      end;
    end;

    IniFile.WriteString('Host' + IntToStr(i+1), 'Hostname', hostname);
    IniFile.WriteInteger('Host' + IntToStr(i+1), 'Timeout', timeout);
    IniFile.WriteBool('Host' + IntToStr(i+1), 'TimeoutDoReset', PingObjs[i].CanReset);

    PingObjs[i].Addr := hostname;
    PingThs[i] := TPingTh.Create(hostname, timeout, i);
  end;
  IniFile.Free;

  for i := 0 to cNumHosts - 1 do
    PingThs[i].Resume;
end;

procedure ProgramEnd;
var
  i: integer;
begin
  for i := 0 to cNumHosts - 1 do
    PingThs[i].Terminate;

  if sem_trywait(semReset) <> 0 then
    ThReset.Terminate;

  for i := 0 to cNumHosts - 1 do
  begin
    PingThs[i].WaitFor;
    PingThs[i].Free;
  end;

  if ThReset <> nil then
  begin
    ThReset.WaitFor;
    ThReset.Free;
  end;

  sem_destroy(semReset);
  sem_destroy(semLog);
  Challenges.Free;
  Answers.Free;
end;

end.
