/*
 * cleopatre/application/p1905_managerd/src/p1905_managerd.c
 *
 * (C) Copyright 2013 MSsar Semiconductor, Inc.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <errno.h>
#include <assert.h>
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <unistd.h> // close function
#include <sys/socket.h>

#ifdef SUPPORT_COREDUMP
#include <sys/resource.h>
#endif
#ifdef SUPPORT_BACKTRACE
#include <execinfo.h>
#endif
#include "p1905_managerd.h"
#include "lldp_message_parse.h"
#include "lldp.h"
#include "cmdu.h"
#ifdef SUPPORT_WIFI
#include "wifi_utils.h"
#endif
#include "cmdu_retry_message.h"
#include "message_wait_queue.h"
#include "os.h"
#include "_1905_lib_io.h"
#include "debug.h"
#include "file_io.h"
#include "topology.h"
#include "eloop.h"
#include "ethernet_layer.h"
#include "netlink_event.h"
#include "worker_task.h"
#include "mapfilter_if.h"

#define MAP_CFG_FILE "/etc/map/1905d.cfg"
#define MAP_WTS_BSS_CFG_FILE "/etc/map/wts_bss_info_config"
#define MAP_WIFI_FILE "/etc/wifi_info.txt"

extern int _1905_interface_init(struct p1905_managerd_ctx *ctx);
extern int _1905_interface_process(struct p1905_managerd_ctx *ctx, struct sockaddr_un* from,
					socklen_t fromlen, char *buf, size_t buf_len, char** reply, size_t *resp_len);
extern void _1905_interface_deinit(struct p1905_managerd_ctx *ctx);
extern void _1905_interface_receive_process(int sock, void *eloop_ctx,
					      void *sock_ctx);
void cmdu_process(int sock, void *eloop_ctx, void *sock_ctx);
void lldp_process(int sock, void *eloop_ctx, void *sock_ctx);
void library_cmd_process(int sock, void *eloop_ctx, void *sock_ctx);
void ctrl_cmd_process(int sock, void *eloop_ctx, void *sock_ctx);
void wapp_cmd_process(int sock, void *eloop_ctx, void *sock_ctx);
int common_process(struct p1905_managerd_ctx *ctx, unsigned char *buf);
void netlink_event_process(int sock, void *eloop_ctx, void *sock_ctx);
void discovery_at_boot_up(void *eloop_data, void *user_ctx);
void worker_task_event_process(int sock, void *eloop_ctx, void *sock_ctx);

#define IEEE802_11_band_2P4G 0x00
#define IEEE802_11_band_5GL  0x01
#define IEEE802_11_band_5GH  0x02

#ifdef CHANGE_BR_AGEING_TIME
#define BRIDGE_AGEING_TIME 150//seconds
#endif

#define P1905_TIMER_INTERVAL_2_SEC 3

#define PERIODIC_TIMER_SEC	5
#define P1905_TIMER_INTERVAL_SEC 1   	/*1 * 5s = 5s*/
#define TOPOLOGY_DISCOVERY_COUNT 11     /*11 * 5s = 55s*/
#define DUMP_INFO_COUNT 700
#define SWITCH_TABLE_DUMP_TIME	45
#define ONE_SECOND_CNT 60

#ifdef SUPPORT_ALME
#define ALME_WAIT_LINK_METRIC_RSP_CNT 2
#endif

unsigned char p1905_multicast_address[6]
                    = { 0x01, 0x80, 0xc2, 0x00, 0x00, 0x13 };
unsigned char nearest_bridge_group_address[6]
                    = { 0x01, 0x80, 0xc2, 0x00, 0x00, 0x0e };
unsigned char dev_send_1905_buf[3072] = {0};
int debug_level = DEBUG_ERROR;
volatile sig_atomic_t timer_expired = 0;
volatile sig_atomic_t exit_signal = 0;
volatile sig_atomic_t hpav_info_has_changed = 0;
#define DISC_RETRY_TIME 3

void ethernet_plug_in_handler(void *context, void *data)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)context;
	struct port_status_info *port_info = (struct port_status_info *)data;
	int j = 0, i = 0;
	if (port_info->status == 0)
		return;
	usleep(20000);
	for (j = 1; j < ctx->itf_number; j++) {
		/*send multicast cmdu to all the ethenet interface */
		if (ctx->itf[j].media_type == IEEE802_3_GROUP) {
			debug(DEBUG_OFF, "send topology discovery with vs info by interface[%s]\n", ctx->itf[j].if_name);
			for(i = 0; i < DISC_RETRY_TIME; i++) {
				ctx->mid++;
				insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
					e_topology_discovery_with_vendor_ie, ctx->mid, ctx->itf[j].if_name, 0);
			}
		}
	}
	eloop_register_timeout(1, 0, discovery_at_boot_up, (void *)ctx, NULL);
}

void ethernet_pull_out_handler(void *context, void *data)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)context;
	struct port_status_info *port_info = (struct port_status_info *)data;
	int i = 0, cnt = 0;

	if (port_info->status == 1)
		return;
	cnt = delete_topo_disc_db_by_port(ctx, port_info->port, NULL);
	if (cnt > 0) {
		report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
			ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
			ctx->service, &ctx->ap_cap_entry.oper_bss_head,
			&ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);
		ctx->mid++;
		for(i = 0; i < ctx->itf_number; i++) {
			insert_cmdu_txq((unsigned char*)p1905_multicast_address,
					(unsigned char*)ctx->p1905_al_mac_addr,
					e_topology_notification, ctx->mid,
					ctx->itf[i].if_name, 0);
#ifdef SUPPORT_CMDU_RELIABLE
			cmdu_reliable_send(ctx, e_topology_notification, ctx->mid, i);
#endif
		}
		/*vlan case to clear switch table of upper ap device*/
		debug(DEBUG_OFF, "clear switch table port [%d]\n", port_info->port);
		eth_layer_clear_switch_table();
	}
	unmask_control_conn_port(ctx, port_info->port);
	delete_topo_response_db_by_port_interface(ctx, port_info->port);
	update_topology_tree(ctx);
}

unsigned char delete_non_p1905_neighbor_dev_info_byport(struct non_p1905_neighbor *non_p1905_dev, unsigned char port)
{
    int i = 0;
	unsigned char changed = 0;
    struct non_p1905_neighbor_info *dev_info, *dev_info_temp;

    for (i = 0; i < ITF_NUM; i++) {
        if (!LIST_EMPTY(&(non_p1905_dev->non_p1905nbr_head))) {
            dev_info = LIST_FIRST(&(non_p1905_dev->non_p1905nbr_head));
            while (dev_info) {
                dev_info_temp = LIST_NEXT(dev_info, non_p1905nbr_entry);
				if (dev_info->port_index == port) {
                	LIST_REMOVE(dev_info, non_p1905nbr_entry);
                	free(dev_info);
					changed = 1;
				}
                dev_info = dev_info_temp;
            }
        }
    }
	return changed;
}

unsigned char delete_non_p1905_neighbor_dev_info_bymac(struct non_p1905_neighbor *non_p1905_dev, unsigned char *mac)
{
    int i = 0;
	unsigned char changed = 0;
    struct non_p1905_neighbor_info *dev_info, *dev_info_temp;

    for (i = 0; i < ITF_NUM; i++) {
        if (!LIST_EMPTY(&(non_p1905_dev->non_p1905nbr_head))) {
            dev_info = LIST_FIRST(&(non_p1905_dev->non_p1905nbr_head));
            while (dev_info) {
                dev_info_temp = LIST_NEXT(dev_info, non_p1905nbr_entry);
				if (!os_memcmp(dev_info->itf_mac_addr, mac, ETH_ALEN)) {
                	LIST_REMOVE(dev_info, non_p1905nbr_entry);
                	free(dev_info);
					changed = 1;
				}
                dev_info = dev_info_temp;
            }
        }
    }
	return changed;
}

struct non_p1905_neighbor_info *get_non_p1905_neighbor_dev_info_bymac(struct non_p1905_neighbor *non_p1905_dev,
	unsigned char *mac, unsigned char port)
{
    struct non_p1905_neighbor_info *dev_info;

	LIST_FOREACH(dev_info, &(non_p1905_dev->non_p1905nbr_head), non_p1905nbr_entry) {
		if (!os_memcmp(dev_info->itf_mac_addr, mac, ETH_ALEN) && dev_info->port_index == port) {
			break;
		}
	}
    return dev_info;
}

void non_1905_neighbor_update(void *eloop_data, void *user_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_data;
	int i = 0, j = 0;
	struct topology_discovery_db *discovery_db = NULL;
	struct topology_response_db *rsp_db = NULL;
	struct ethernet_client_entry *client_entry = NULL;
	struct non_p1905_neighbor *non_1905_neighbor_list = NULL;
	struct non_p1905_neighbor_info *info = NULL;
	unsigned char topo_changed = 0;
	struct ethernet_port *peth_data = NULL;
	struct non_p1905_neighbor_info *dev_info, *dev_info_temp;

	if (eth_layer_port_client_update() < 0)
		return;

	for (j = FIRST_VITUAL_ITF + 1; j < ctx->itf_number; j++) {
		if (ctx->itf[j].media_type != IEEE802_3_GROUP)
			continue;

		non_1905_neighbor_list = &ctx->non_p1905_neighbor_dev[j];

		for (i = 0; i < MAX_LAN_PORT_NUM; i++) {
			discovery_db = NULL;
			peth_data = eth_layer_get_eth_data_by_port_num(i);
			if(peth_data == NULL) {
				debug(DEBUG_ERROR, "peth_data null\n");
				return;
			}
			dl_list_for_each(client_entry, &peth_data->clients, struct ethernet_client_entry, entry) {
				discovery_db = get_tpd_db_by_mac(ctx, client_entry->mac);

				/*to check any 1905 device exist*/
				if (discovery_db != NULL) {
					/*delete all non-1905 neighbor device belongs to this port*/
					debug(DEBUG_TRACE, "delete all non 1905 neighbors for 1905 topology discovery on port %d\n",
						peth_data->port_index);
					topo_changed = delete_non_p1905_neighbor_dev_info_byport(non_1905_neighbor_list, peth_data->port_index);
					break;
				}
			}
			/*if no direct 1905 device found with current port*/
			if (discovery_db == NULL) {

				dl_list_for_each(client_entry, &peth_data->clients, struct ethernet_client_entry, entry) {
					rsp_db = get_trsp_db_by_mac(ctx, client_entry->mac);
					if (rsp_db != NULL) {
						break;
					}
				}
				/*if this almac is found in topology response db, should not add it into non 1905 neighbor list*/
				if (rsp_db == NULL) {
					dl_list_for_each(client_entry, &peth_data->clients, struct ethernet_client_entry, entry) {
						info = get_non_p1905_neighbor_dev_info_bymac(non_1905_neighbor_list, client_entry->mac, peth_data->port_index);
						if (info == NULL) {
							info = (struct non_p1905_neighbor_info*)os_malloc(sizeof(struct non_p1905_neighbor_info));
							if (info == NULL)
								continue;
							info->port_index = peth_data->port_index;
							os_memcpy(info->itf_mac_addr, client_entry->mac, ETH_ALEN);
							debug(DEBUG_TRACE, "insert non 1905 device %02x:%02x:%02x:%02x:%02x:%02x on inf %02x:%02x:%02x:%02x:%02x:%02x port %d\n",
								PRINT_MAC(info->itf_mac_addr), PRINT_MAC(non_1905_neighbor_list->local_mac_addr),
								peth_data->port_index);
							LIST_INSERT_HEAD(&non_1905_neighbor_list->non_p1905nbr_head, info, non_p1905nbr_entry);
							topo_changed = 1;
						}
					}
				} else {
					topo_changed = delete_non_p1905_neighbor_dev_info_bymac(non_1905_neighbor_list, rsp_db->al_mac_addr);
				}

				/*delete the non 1905 neighbor from list by checking the switch port data list*/
				dev_info = LIST_FIRST(&(non_1905_neighbor_list->non_p1905nbr_head));
	            while (dev_info) {
	                dev_info_temp = LIST_NEXT(dev_info, non_p1905nbr_entry);
					if (dev_info->port_index == peth_data->port_index) {
						if (!eth_layer_get_client_by_mac(peth_data, dev_info->itf_mac_addr)) {
							LIST_REMOVE(dev_info, non_p1905nbr_entry);
							debug(DEBUG_TRACE, "delete non 1905 device %02x:%02x:%02x:%02x:%02x:%02x on inf %02x:%02x:%02x:%02x:%02x:%02x port %d\n",
														PRINT_MAC(dev_info->itf_mac_addr), PRINT_MAC(non_1905_neighbor_list->local_mac_addr),
														dev_info->port_index);
		                	os_free(dev_info);
							topo_changed = 1;
						}
					}
	                dev_info = dev_info_temp;
	            }
			}
		}
	}
	if (topo_changed) {
		ctx->mid++;
		for (j = 0; j < ctx->itf_number; j++) {
			/*send multicast cmdu to all the interface */
			debug(DEBUG_TRACE, "send notification from %02x:%02x:%02x:%02x:%02x:%02x for discovery\n",
				PRINT_MAC(ctx->p1905_al_mac_addr));
			insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
				e_topology_notification, ctx->mid, ctx->itf[j].if_name, 0);
#ifdef SUPPORT_CMDU_RELIABLE
			cmdu_reliable_send(ctx, e_topology_notification, ctx->mid, j);
#endif
		}
	}

	eloop_register_timeout(SWITCH_TABLE_DUMP_TIME, 0, non_1905_neighbor_update, (void *)ctx, NULL);
}


#define CONFIG_SYNC_TOLERATE_TIME 30
#define RENEW_ROUND 4
void config_sync_error_handler(void *context, void* parent_leaf, void *current_leaf, void *data)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)context;
	struct leaf *cur_leaf = (struct leaf *)current_leaf;
	struct os_time now;
	struct topology_response_db *rpdb = NULL;
	unsigned char band = 0;
	struct agent_list_db *agent = NULL;

	os_get_time(&now);


	find_agent_info(ctx, cur_leaf->al_mac, &agent);
	if (agent) {
		cur_leaf->band_cap |= agent->band_cap;
		cur_leaf->config_status |= agent->band_configured;
	}

	/*it means only after autoconfig, can check whether should renew the deivce*/
	/*
	if (cur_leaf->band_cap == 0)
		return;
	*/
	if (now.sec - cur_leaf->create_time.sec > CONFIG_SYNC_TOLERATE_TIME) {
		/*to let send renew per about 20 seconds if peer device still can not be configured*/
		if (cur_leaf->renew_send_round > 0)
			cur_leaf->renew_send_round--;

		if (cur_leaf->renew_send_round == 0) {

			rpdb = lookup_tprdb_by_almac(ctx, cur_leaf->al_mac);
			if (!rpdb) {
				debug(DEBUG_ERROR, "BUG here!!!! rpdb almac(%02x:%02x:%02x:%02x:%02x:%02x) not exist\n",
					PRINT_MAC(cur_leaf->al_mac));
				return;
			}
			cur_leaf->renew_send_round = RENEW_ROUND;

			/*it means the agent did not start autoconfiguration within 30 seconds after joining network*/
			if (agent == NULL || (agent && agent->band_cap == 0)) {
				/*could not get peer's band capability, assume it could support both 2.4G and 5G band*/
				cur_leaf->band_cap |= (BAND_2G_CAP | BAND_5G_CAP);
			}

			if ((cur_leaf->band_cap & BAND_2G_CAP) && !(cur_leaf->config_status & CONFIGURED_2G)) {
				band = 0;
				if (fill_send_tlv(ctx, &band, 1) < 0) {
					return;
				}
				debug(DEBUG_ERROR, "config push to almac(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(cur_leaf->al_mac));
				debug(DEBUG_ERROR, "\t renew band %s\n", band == 0 ? "2G" : "5G");
				insert_cmdu_txq(cur_leaf->al_mac, ctx->p1905_al_mac_addr,
						e_ap_autoconfiguration_renew, ++ctx->mid, (rpdb->recv_ifid != -1) ? ctx->itf[rpdb->recv_ifid].if_name : ctx->br_name, 0);
				process_cmdu_txq(ctx, ctx->rx_buf);
			}

			if ((cur_leaf->band_cap & BAND_5G_CAP) && !(cur_leaf->config_status & CONFIGURED_5G)) {
				band = 1;
				if (fill_send_tlv(ctx, &band, 1) < 0) {
					return;
				}
				debug(DEBUG_ERROR, "config push to almac(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(cur_leaf->al_mac));
				debug(DEBUG_ERROR, "\t renew band %s\n", band == 0 ? "2G" : "5G");
				insert_cmdu_txq(cur_leaf->al_mac, ctx->p1905_al_mac_addr,
						e_ap_autoconfiguration_renew, ++ctx->mid, (rpdb->recv_ifid != -1) ? ctx->itf[rpdb->recv_ifid].if_name : ctx->br_name, 0);
				process_cmdu_txq(ctx, ctx->rx_buf);
			}
		}
	}
}

void manage_cmd_process(struct p1905_managerd_ctx *ctx, struct manage_cmd *cmd)
{
	unsigned char buf[512] = {0};
	size_t len = 0;
	unsigned short length = 0;
	switch(cmd->cmd_id) {
		case MANAGE_SET_ROLE:
			{
				if (change_role_dynamic(ctx, cmd->content[0]) < 0) {
					os_strncpy((char*)buf, "FAIL\n", 5);
					len = 5;
				} else {
					os_strncpy((char*)buf, "OK\n", 3);
					len = 3;
				}
				_1905_notify_raw_data(ctx, buf, len);
			}
			break;
		case MANAGE_SET_LOG_LEVEL:
			break;
		case MANAGE_GET_LOCAL_DEVINFO:
			if (get_local_device_info(ctx, buf, &length) < 0) {
				debug(DEBUG_ERROR, "error when get device info\n");
				break;
			}
			len = length;
			_1905_notify_raw_data(ctx, buf, len);
			eloop_register_timeout(1,0, attach_action, (void *)ctx, NULL);
			break;
		case MANAGE_SET_BSS_CONF:
			{
				unsigned char oper = cmd->content[0];
				set_bss_config(ctx, oper, (struct bss_config_info*)(&cmd->content[1]));
			}
			break;
		default:
			break;
	}

}

void p1905_manage_terminate(int signum, void *signal_ctx)
{
	eloop_terminate();
}

#define NEIGHBOR_TOPOLOGY_QUERY_TIME 20
#define NEIGHBOR_TOPOLOGY_EXPIRE_TIME 40
#define TOPOLOGY_EXPIRE_TIME 70
#define TOPOLOGY_EXPIRE_DEL_TIME 90
void delete_exist_topology_response_database(struct p1905_managerd_ctx *ctx,
                                                unsigned char *al_mac);

static void check_topology_rsp_expired(struct p1905_managerd_ctx *ctx)
{
	struct os_time now;
	struct topology_response_db *rsp, *rsp_tmp = NULL;
	struct topology_discovery_db *disc = NULL, *disc_tmp = NULL;
	char query = 0, del = 0, del_rsp_cnt = 0;
	int i = 0;

	os_get_time(&now);

	/*fistly,  check whether it is necessary query all devices in topology response database*/
	disc = LIST_FIRST(&ctx->topology_entry.tpddb_head);
	while (disc) {
		query = 0;
		del = 0;
		disc_tmp = LIST_NEXT(disc, tpddb_entry);
		rsp = find_response_by_almac(ctx, disc->al_mac);
		if (rsp) {
			if ((now.sec - rsp->last_seen.sec) > NEIGHBOR_TOPOLOGY_EXPIRE_TIME) {
				del = 1;
			} else if ((now.sec - rsp->last_seen.sec) > NEIGHBOR_TOPOLOGY_QUERY_TIME) {
				query = 1;
			}
		} else {
			query = 1;
		}

		if (query) {
			/*get ifname by discovery*/
			for (i = 0; i < ctx->itf_number; i++) {
				if (!os_memcmp(ctx->itf[i].mac_addr, disc->receive_itf_mac, ETH_ALEN))
					break;
			}
			if (i < ctx->itf_number)
				insert_cmdu_txq(disc->al_mac, ctx->p1905_al_mac_addr,
					e_topology_query, ++ctx->mid, ctx->itf[i].if_name, 0);
			else
				insert_cmdu_txq(disc->al_mac, ctx->p1905_al_mac_addr,
					e_topology_query, ++ctx->mid, ctx->br_name, 0);
			debug(DEBUG_OFF, "send query(%04x) to "
				"neighbor(%02x:%02x:%02x:%02x:%02x:%02x) "
				"by ifname(%s)\n", ctx->mid, PRINT_MAC(disc->al_mac),
				(i < ctx->itf_number) ? ctx->itf[i].if_name : ctx->br_name);
		}
		if (del) {
			delete_p1905_neighbor_dev_info(ctx, disc->al_mac, disc->receive_itf_mac);
			report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
				ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
				ctx->service, &ctx->ap_cap_entry.oper_bss_head,
				&ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);
			LIST_REMOVE(disc, tpddb_entry);
			if (find_discovery_by_almac(ctx, disc->al_mac) == NULL) {
				delete_exist_topology_response_database(ctx, disc->al_mac);
				del_rsp_cnt++;
				debug(DEBUG_OFF, "no discovery in data base; del rsp "
					"(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(disc->al_mac));
			}
			unmask_control_conn_port(ctx, disc->eth_port);
			free(disc);
		}
		disc = disc_tmp;
	}

	rsp = SLIST_FIRST(&ctx->topology_entry.tprdb_head);
	while (rsp) {
		query = 0;
		del = 0;
		rsp_tmp = SLIST_NEXT(rsp, tprdb_entry);
		if (find_discovery_by_almac(ctx, rsp->al_mac_addr) == NULL) {
			if ((now.sec - rsp->last_seen.sec > TOPOLOGY_EXPIRE_TIME) &&
				(now.sec - rsp->last_seen.sec < TOPOLOGY_EXPIRE_DEL_TIME) ) {
				query = 1;
			} else if (now.sec - rsp->last_seen.sec > TOPOLOGY_EXPIRE_DEL_TIME) {
				del = 1;
			}
		}

		if (query) {
			debug(DEBUG_WARN, "need send query to (%02x:%02x:%02x:%02x:%02x:%02x)\n"
				, PRINT_MAC(rsp->al_mac_addr));
			insert_cmdu_txq(rsp->al_mac_addr, ctx->p1905_al_mac_addr,
				e_topology_query, ++ctx->mid,
				(rsp->recv_ifid != -1) ? ctx->itf[rsp->recv_ifid].if_name : ctx->br_name, 0);
		}
		if (del) {
			debug(DEBUG_OFF, "delete non neighbor rsp(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(rsp->al_mac_addr));
			delete_exist_topology_response_database(ctx, rsp->al_mac_addr);
			del_rsp_cnt++;
		}
		rsp = rsp_tmp;
	}

	/*secondly, check whether it is necessary to update my own topology response*/
	if (now.sec - ctx->own_topo_rsp_update_time.sec > TOPOLOGY_EXPIRE_TIME) {
		debug(DEBUG_WARN, "my own topology rsp update sec%ld\n", ctx->own_topo_rsp_update_time.sec);
		report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
			ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
			ctx->service, &ctx->ap_cap_entry.oper_bss_head,
			&ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);
	}

	if (del_rsp_cnt)
		mark_valid_topo_rsp_node(ctx);
}


/*1905 one sec timer*/
void p1905_managerd_periodic(void *eloop_data, void *user_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx *)eloop_data;
	static unsigned int cnt = 0;
	int i = 0;
	unsigned char *buf = ctx->rx_buf;

	cnt += PERIODIC_TIMER_SEC;

    /*1. update LLDP database, if exceed time to live , delete*/
    update_lldp_queue_ttl(PERIODIC_TIMER_SEC);

    /*2. delete 1905.1 neighbor if not receiving new topology discovery*/
    if (delete_not_exist_p1905_neighbor_device(ctx, PERIODIC_TIMER_SEC)) {
    	ctx->mid++;
		for(i = 0; i < ctx->itf_number; i++) {
        	insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
                        e_topology_notification, ctx->mid, ctx->itf[i].if_name, 0);
#ifdef SUPPORT_CMDU_RELIABLE
			cmdu_reliable_send(ctx, e_topology_notification, ctx->mid, i);
#endif
		}
    }
	/*3 delete 1905 topology device*/
	delete_not_exist_p1905_topology_device(ctx, PERIODIC_TIMER_SEC);
	check_topology_rsp_expired(ctx);
	/*update topology tree*/
	update_topology_tree(ctx);
	/*config sync error handling*/
	if (ctx->role == CONTROLLER)
		trace_topology_tree_cb(ctx, ctx->root_leaf, (topology_tree_cb_func)config_sync_error_handler, NULL);

	if ((cnt % ONE_SECOND_CNT) == 0) {
        /*increment fragment timeout counter*/
        add_fragment_cnt();
        if (get_fragment_cnt() > 2)
            delete_fragment_queue_all();
    }

	/*check if any message need resend*/
	handle_retry_message_queue(ctx);
	common_process(ctx, buf);
	eloop_register_timeout(PERIODIC_TIMER_SEC, 0, p1905_managerd_periodic, (void *)ctx, NULL);
}

void send_discovery_periodic(void *eloop_data, void *user_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx *)eloop_data;
	int i = 0;
	unsigned char *buf = ctx->rx_buf;

	ctx->mid++;
	for (i = 0; i < ctx->itf_number; i++)
		insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
			e_topology_discovery, ctx->mid, ctx->itf[i].if_name, 0);

     /*according to IEEE P1905.1 spec, we need to send 802.1 bridge
         *detection message after sending topology discovery message
         *so we implement this functionality here
         */
	for(i = FIRST_VITUAL_ITF + 1; i < ctx->itf_number; i++) {
		if(0 > send_802_1_bridge_discovery_msg(ctx,
			nearest_bridge_group_address, ctx->itf[i].mac_addr, buf, i))
			debug(DEBUG_ERROR, "send lldpdu fail on %s\n", ctx->itf[i].if_name);
	}

	common_process(ctx, buf);
	eloop_register_timeout(ctx->discovery_cnt * PERIODIC_TIMER_SEC, 0,
		send_discovery_periodic, (void *)ctx, NULL);
}


void p1905_database_init(struct p1905_managerd_ctx *ctx)
{
	ctx->service = ctx->role;
	ctx->mid = (unsigned short)os_random();
    LIST_INIT(&ctx->topology_entry.tpddb_head);
    SLIST_INIT(&ctx->topology_entry.tprdb_head);
    SLIST_INIT(&ctx->topodev_entry.tpdevdb_head);
	SLIST_INIT(&ctx->ap_cap_entry.ap_ht_cap_head);
	SLIST_INIT(&ctx->ap_cap_entry.ap_vht_cap_head);
	SLIST_INIT(&ctx->ap_cap_entry.basic_cap_head);
	SLIST_INIT(&ctx->ap_cap_entry.oper_bss_head);
	SLIST_INIT(&ctx->ap_cap_entry.assoc_clients_head);
	SLIST_INIT(&ctx->ap_cap_entry.ch_prefer_head);
	SLIST_INIT(&ctx->ap_cap_entry.oper_restrict_head);
	SLIST_INIT(&ctx->map_policy.spolicy.local_disallow_head);
	SLIST_INIT(&ctx->map_policy.spolicy.btm_disallow_head);
	SLIST_INIT(&ctx->map_policy.spolicy.radio_policy_head);
	SLIST_INIT(&ctx->map_policy.mpolicy.policy_head);
	SLIST_INIT(&ctx->metric_entry.metrics_query_head);
	SLIST_INIT(&ctx->metric_entry.metrics_rsp_head);
	SLIST_INIT(&ctx->metric_entry.traffic_stats_head);
	SLIST_INIT(&ctx->metric_entry.link_metrics_head);
	SLIST_INIT(&ctx->metric_entry.unlink_info.unlink_metrics_head);
	SLIST_INIT(&ctx->agent_head);
	SLIST_INIT(&ctx->query_neighbor_head);
#ifdef MAP_R2
	SLIST_INIT(&ctx->ap_cap_entry.ch_scan_cap_head);
	SLIST_INIT(&ctx->ap_cap_entry.radio_cac_cap.radio_cac_capab_head);
	SLIST_INIT(&ctx->metric_entry.radio_metrics_head);
	SLIST_INIT(&ctx->metric_entry.assoc_sta_extended_link_metrics_head);
	SLIST_INIT(&ctx->metric_entry.radio_identifier_head);
	SLIST_INIT(&ctx->map_policy.tpolicy.policy_head);
	SLIST_INIT(&ctx->map_policy.epolicy.policy_head);
	SLIST_INIT(&ctx->map_policy.fpolicy.bssid_head);
#endif
}

extern 	int debug_level;

void clear_topology_db(struct p1905_topology_db *topo_entry)
{
	struct topology_discovery_db *pddb = NULL, *pddb_tmp = NULL;
	struct topology_response_db *prdb = NULL, *prdb_tmp = NULL;
	struct device_info_db *pdev_info_db = NULL, *pdev_info_db_tmp = NULL;
	struct p1905_neighbor_device_db *neighbor_dev_db = NULL, *neighbor_dev_db_tmp = NULL;
	struct non_p1905_neighbor_device_list_db *non_neighbor_dev_db = NULL;
	struct non_p1905_neighbor_device_list_db *non_neighbor_dev_db_tmp = NULL;
	struct device_bridge_capability_db *pbrcap_db = NULL, *pbrcap_db_tmp = NULL;

	/*free list tpddb_head*/
	pddb = LIST_FIRST(&topo_entry->tpddb_head);
	while (pddb != NULL) {
		pddb_tmp = LIST_NEXT(pddb, tpddb_entry);
		debug(DEBUG_TRACE, "free topology_discovery_db(al=%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(pddb->al_mac));
		free(pddb);
		pddb = pddb_tmp;
	}
	LIST_INIT(&topo_entry->tpddb_head);

	/*free list tprdb_head*/
	prdb = SLIST_FIRST(&topo_entry->tprdb_head);
	while (prdb != NULL) {
		prdb_tmp = SLIST_NEXT(prdb, tprdb_entry);
		debug(DEBUG_TRACE, "[%s]free topology_response_db(al=%02x:%02x:%02x:%02x:%02x:%02x)\n",
			__func__, PRINT_MAC(prdb->al_mac_addr));

		/*free devinfo_head*/
		pdev_info_db = SLIST_FIRST(&prdb->devinfo_head);
		while (pdev_info_db != NULL) {
			pdev_info_db_tmp = SLIST_NEXT(pdev_info_db, devinfo_entry);
			debug(DEBUG_TRACE, "[%s]free device_info_db(mac=%02x:%02x:%02x:%02x:%02x:%02x)\n",
				__func__, PRINT_MAC(pdev_info_db->mac_addr));

			/*free p1905_neighbor_device_db*/
			neighbor_dev_db = SLIST_FIRST(&pdev_info_db->p1905_nbrdb_head);
			while (neighbor_dev_db != NULL) {
				neighbor_dev_db_tmp = SLIST_NEXT(neighbor_dev_db, p1905_nbrdb_entry);
				debug(DEBUG_TRACE, "[%s]free neighbor_dev_db(al=%02x:%02x:%02x:%02x:%02x:%02x)\n",
					__func__, PRINT_MAC(neighbor_dev_db->p1905_neighbor_al_mac));
				free(neighbor_dev_db);
				neighbor_dev_db = neighbor_dev_db_tmp;
			}
			SLIST_INIT(&pdev_info_db->p1905_nbrdb_head);

			/*free non_p1905_neighbor_device_list_db*/
			non_neighbor_dev_db = SLIST_FIRST(&pdev_info_db->non_p1905_nbrdb_head);
			while (non_neighbor_dev_db != NULL) {
				non_neighbor_dev_db_tmp = SLIST_NEXT(non_neighbor_dev_db, non_p1905_nbrdb_entry);
				debug(DEBUG_TRACE, "[%s]free non neighbor_dev_db(al=%02x:%02x:%02x:%02x:%02x:%02x)\n",
					__func__, PRINT_MAC(non_neighbor_dev_db->non_p1905_device_interface_mac));
				free(non_neighbor_dev_db);
				non_neighbor_dev_db = non_neighbor_dev_db_tmp;
			}
			SLIST_INIT(&(pdev_info_db->non_p1905_nbrdb_head));

			if(pdev_info_db->vs_info_len)
				free(pdev_info_db->vs_info);

			free(pdev_info_db);
			pdev_info_db = pdev_info_db_tmp;
		}
		SLIST_INIT(&prdb->devinfo_head);

		/*free device_bridge_capability_db*/
		pbrcap_db = LIST_FIRST(&prdb->brcap_head);
		while (pbrcap_db != NULL) {
			pbrcap_db_tmp = LIST_NEXT(pbrcap_db, brcap_entry);
			if(pbrcap_db->interface_amount != 0)
				free(pbrcap_db->interface_mac_tuple);
			free(pbrcap_db);
			pbrcap_db = pbrcap_db_tmp;
		}
		LIST_INIT(&(prdb->brcap_head));

		free(prdb);
		prdb = prdb_tmp;
	}
	SLIST_INIT(&topo_entry->tprdb_head);

}

void clear_topology_neighbor_db(struct p1905_neighbor *neighbor, unsigned char itf_number)
{
	struct p1905_neighbor_info *pinfo = NULL, *pinfo_tmp = NULL;
	int i = 0;

	for(i = 0; i < itf_number; i++) {
		pinfo = LIST_FIRST(&neighbor[i].p1905nbr_head);
		while (pinfo != NULL) {
			pinfo_tmp = LIST_NEXT(pinfo, p1905nbr_entry);
			debug(DEBUG_TRACE, "[%s]neighbor[%d]free p1905_neighbor_info(al=%02x:%02x:%02x:%02x:%02x:%02x)\n",
				__func__,i, PRINT_MAC(pinfo->al_mac_addr));
			free(pinfo);
			pinfo = pinfo_tmp;
		}
		LIST_INIT(&neighbor[i].p1905nbr_head);
	}
}

void clear_topology_device_db(struct p1905_topodevice_db *topodev_entry)
{
	struct topology_device_db *pdev = NULL, *pdev_tmp = NULL;

	pdev = SLIST_FIRST(&topodev_entry->tpdevdb_head);
	while (pdev != NULL) {
		pdev_tmp = SLIST_NEXT(pdev, tpdev_entry);
		debug(DEBUG_TRACE, "[%s]free topology device(al=%02x:%02x:%02x:%02x:%02x:%02x)\n",
			__func__, PRINT_MAC(pdev->aldev_mac));
		free(pdev);
		pdev = pdev_tmp;
	}
	SLIST_INIT(&topodev_entry->tpdevdb_head);
}

void clear_ap_capability_db(struct ap_capability_db *apcap_entry)
{
	struct ap_ht_cap_db *pht_cap_db = NULL, *pht_cap_db_tmp = NULL;
	struct ap_vht_cap_db *pvht_cap_db = NULL, *pvht_cap_db_tmp = NULL;
	struct radio_basic_capability_db *pbasic_cap_db = NULL, *pbasic_cap_db_tmp = NULL;
	struct basic_cap_db *cap_db = NULL, *cap_db_tmp = NULL;
	struct operational_bss_db *oper_bss_db = NULL, *oper_bss_db_tmp = NULL;
	struct op_bss_db *bss_db = NULL, *bss_db_tmp = NULL;
	struct associated_clients_db *assoc_client_db = NULL, *assoc_client_db_tmp = NULL;
	struct clients_db *cli_db = NULL, *cli_db_tmp = NULL;
	struct ch_prefer_db *ch_pre_db	= NULL, *ch_pre_db_tmp = NULL;
	struct prefer_info_db *pre_db = NULL, *pre_db_tmp = NULL;
	struct oper_restrict_db *restriction  = NULL, *restriction_tmp = NULL;
	struct restrict_db *resdb = NULL, *resdb_tmp = NULL;

	/*free ap_ht_cap_head*/
	pht_cap_db = SLIST_FIRST(&apcap_entry->ap_ht_cap_head);
	while (pht_cap_db != NULL) {
		pht_cap_db_tmp = SLIST_NEXT(pht_cap_db, ap_ht_cap_entry);
		debug(DEBUG_TRACE, "free ht_cap_db(id=%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(pht_cap_db->identifier));
		free(pht_cap_db);
		pht_cap_db = pht_cap_db_tmp;
	}
	SLIST_INIT(&apcap_entry->ap_ht_cap_head);

	/*free ap_vht_cap_head*/
	pvht_cap_db = SLIST_FIRST(&apcap_entry->ap_vht_cap_head);
	while (pvht_cap_db != NULL) {
		pvht_cap_db_tmp = SLIST_NEXT(pvht_cap_db, ap_vht_cap_entry);
		debug(DEBUG_TRACE, "free vht_cap_db(id=%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(pvht_cap_db->identifier));
		free(pvht_cap_db);
		pvht_cap_db = pvht_cap_db_tmp;
	}
	SLIST_INIT(&apcap_entry->ap_vht_cap_head);

	/*free basic_cap_head*/
	pbasic_cap_db = SLIST_FIRST(&apcap_entry->basic_cap_head);
	while (pbasic_cap_db != NULL) {
		pbasic_cap_db_tmp = SLIST_NEXT(pbasic_cap_db, bcap_entry);
		debug(DEBUG_TRACE, "free basic_cap_db(id=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(pbasic_cap_db->identifier));
		/*free basic_cap_db*/
		cap_db = SLIST_FIRST(&pbasic_cap_db->bcap_head);
		while (cap_db != NULL) {
			cap_db_tmp = SLIST_NEXT(cap_db, basic_cap_entry);
			debug(DEBUG_TRACE, "free cap_db oper_class=%d\n", cap_db->op_class);
			free(cap_db);
			cap_db = cap_db_tmp;
		}
		SLIST_INIT(&pbasic_cap_db->bcap_head);
		free(pbasic_cap_db);
		pbasic_cap_db = pbasic_cap_db_tmp;
	}
	SLIST_INIT(&apcap_entry->basic_cap_head);

	/*free oper_bss_head*/
	oper_bss_db = SLIST_FIRST(&apcap_entry->oper_bss_head);
	while (oper_bss_db != NULL) {
		oper_bss_db_tmp = SLIST_NEXT(oper_bss_db, oper_bss_entry);
		debug(DEBUG_TRACE, "free operational_bss_db(id=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(oper_bss_db->identifier));
		/*free basic_cap_db*/
		bss_db = SLIST_FIRST(&oper_bss_db->op_bss_head);
		while (bss_db != NULL) {
			bss_db_tmp = SLIST_NEXT(bss_db, op_bss_entry);
			debug(DEBUG_TRACE, "free bss_db bssid[%02x:%02x:%02x:%02x:%02x:%02x]\n",
				PRINT_MAC(bss_db->bssid));
			free(bss_db);
			bss_db = bss_db_tmp;
		}
		SLIST_INIT(&oper_bss_db->op_bss_head);
		free(oper_bss_db);
		oper_bss_db = oper_bss_db_tmp;
	}
	SLIST_INIT(&apcap_entry->oper_bss_head);

	/*free assoc_clients_head*/
	assoc_client_db = SLIST_FIRST(&apcap_entry->assoc_clients_head);
	while (assoc_client_db != NULL) {
		assoc_client_db_tmp = SLIST_NEXT(assoc_client_db, assoc_clients_entry);
		debug(DEBUG_TRACE, "free associated_clients_db(bssid=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(assoc_client_db->bssid));
		/*free basic_cap_db*/
		cli_db = SLIST_FIRST(&assoc_client_db->clients_head);
		while (cli_db != NULL) {
			cli_db_tmp = SLIST_NEXT(cli_db, clients_entry);
			debug(DEBUG_TRACE, "free cli_db mac[%02x:%02x:%02x:%02x:%02x:%02x]\n",
				PRINT_MAC(cli_db->mac));
			free(cli_db);
			cli_db = cli_db_tmp;
		}
		SLIST_INIT(&assoc_client_db->clients_head);
		free(assoc_client_db);
		assoc_client_db = assoc_client_db_tmp;
	}
	SLIST_INIT(&apcap_entry->assoc_clients_head);

	/*free ch_prefer_head*/
	ch_pre_db = SLIST_FIRST(&apcap_entry->ch_prefer_head);
	while (ch_pre_db != NULL) {
		ch_pre_db_tmp = SLIST_NEXT(ch_pre_db, ch_prefer_entry);
		debug(DEBUG_TRACE, "free prefer_info_entry(id=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(ch_pre_db->identifier));
		/*free pre_db*/
		pre_db = SLIST_FIRST(&ch_pre_db->prefer_info_head);
		while (pre_db != NULL) {
			pre_db_tmp = SLIST_NEXT(pre_db, prefer_info_entry);
			debug(DEBUG_TRACE, "free pre_db oper_class=%d\n", pre_db->op_class);
			free(pre_db);
			pre_db = pre_db_tmp;
		}
		SLIST_INIT(&ch_pre_db->prefer_info_head);
		free(ch_pre_db);
		ch_pre_db = ch_pre_db_tmp;
	}
	SLIST_INIT(&apcap_entry->ch_prefer_head);

	/*free oper_restrict_head*/
	restriction = SLIST_FIRST(&apcap_entry->oper_restrict_head);
	while (restriction != NULL) {
		restriction_tmp = SLIST_NEXT(restriction, oper_restrict_entry);
		debug(DEBUG_TRACE, "free oper_restrict_db(id=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(restriction->identifier));
		/*free restrict_db*/
		resdb = SLIST_FIRST(&restriction->restrict_head);
		while (resdb != NULL) {
			resdb_tmp = SLIST_NEXT(resdb, restrict_entry);
			debug(DEBUG_TRACE, "free pre_db oper_class=%d\n", resdb->op_class);
			free(resdb);
			resdb = resdb_tmp;
		}
		SLIST_INIT(&restriction->restrict_head);
		free(restriction);
		restriction = restriction_tmp;
	}
	SLIST_INIT(&apcap_entry->oper_restrict_head);

	apcap_entry->ap_cap.rssi_steer = 0;
	apcap_entry->ap_cap.sta_report_not_cop = 0;
	apcap_entry->ap_cap.sta_report_on_cop = 0;
}

void clear_metric_db(struct metrics_info *metrics_entry)
{
	struct bss_db *bss = NULL, *bss_tmp = NULL;
	struct mrsp_db *mrsp = NULL, *mrsp_tmp = NULL;
	struct esp_db *esp = NULL, *esp_tmp = NULL;
	struct traffic_stats_db *traffic_stats = NULL, *traffic_stats_tmp = NULL;
	struct stats_db *stats = NULL, *stats_tmp = NULL;
	struct link_metrics_db *link_metrics = NULL, *link_metrics_tmp = NULL;
	struct metrics_db *metrics = NULL, *metrics_tmp = NULL;

	/*metrics query*/
	bss = SLIST_FIRST(&metrics_entry->metrics_query_head);
	while (bss != NULL) {
		bss_tmp = SLIST_NEXT(bss, bss_entry);
		debug(DEBUG_TRACE, "free bss_db(id=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(bss->bssid));
		free(bss);
		bss = bss_tmp;
	}
	SLIST_INIT(&metrics_entry->metrics_query_head);
	metrics_entry->metrics_query_cnt = 0;

	/*metrics response*/
	mrsp = SLIST_FIRST(&metrics_entry->metrics_rsp_head);
	while (mrsp != NULL) {
		mrsp_tmp = SLIST_NEXT(mrsp, mrsp_entry);
		debug(DEBUG_TRACE, "free mrsp_db(id=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(mrsp->bssid));
		/*free esp_db*/
		esp = SLIST_FIRST(&mrsp->esp_head);
		while (esp != NULL) {
			esp_tmp = SLIST_NEXT(esp, esp_entry);
			debug(DEBUG_TRACE, "free esp_db ac=%d\n", esp->ac);
			free(esp);
			esp = esp_tmp;
		}
		SLIST_INIT(&mrsp->esp_head);
		free(mrsp);
		mrsp = mrsp_tmp;
	}
	SLIST_INIT(&metrics_entry->metrics_rsp_head);
	metrics_entry->metrics_rsp_cnt = 0;

	/*associated sta traffic stats*/
	traffic_stats = SLIST_FIRST(&metrics_entry->traffic_stats_head);
	while (traffic_stats != NULL) {
		traffic_stats_tmp = SLIST_NEXT(traffic_stats, traffic_stats_entry);
		debug(DEBUG_TRACE, "free traffic_stats_db(id=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(traffic_stats->identifier));
		/*free stats_db*/
		stats = SLIST_FIRST(&traffic_stats->stats_head);
		while (stats != NULL) {
			stats_tmp = SLIST_NEXT(stats, stats_entry);
			debug(DEBUG_TRACE, "free stats_db bytes_sent=%d\n", stats->bytes_sent);
			free(stats);
			stats = stats_tmp;
		}
		SLIST_INIT(&traffic_stats->stats_head);
		free(traffic_stats);
		traffic_stats = traffic_stats_tmp;
	}
	SLIST_INIT(&metrics_entry->traffic_stats_head);

	/*associated sta link metrics*/
	link_metrics = SLIST_FIRST(&metrics_entry->link_metrics_head);
	while(link_metrics != NULL) {
		link_metrics_tmp = SLIST_NEXT(link_metrics, link_metrics_entry);
		debug(DEBUG_TRACE, "free link_metrics_db(id=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(link_metrics->identifier));
		/*free metrics_db*/
		metrics = SLIST_FIRST(&link_metrics->metrics_head);
		while (metrics != NULL) {
			metrics_tmp = SLIST_NEXT(metrics, metrics_entry);
			debug(DEBUG_TRACE, "free metrics_db erate_downlink=%d\n", metrics->erate_downlink);
			free(metrics);
			metrics = metrics_tmp;
		}
		SLIST_INIT(&link_metrics->metrics_head);
		free(link_metrics);
		link_metrics = link_metrics_tmp;
	}
	SLIST_INIT(&metrics_entry->link_metrics_head);

	memset(metrics_entry->assoc_sta, 0, ETH_ALEN);
	memset(&metrics_entry->assoc_sta_link_metrics, 0, sizeof(struct metrics_db));

	/*unassociated sta link metrics*/
	delete_exist_unlink_metrics_rsp(&metrics_entry->unlink_info);
	if (metrics_entry->unlink_query) {
		free(metrics_entry->unlink_query);
		metrics_entry->unlink_query = NULL;
	}
	if (metrics_entry->bcn_query) {
		free(metrics_entry->bcn_query);
		metrics_entry->bcn_query = NULL;
	}
	if (metrics_entry->bcn_rsp) {
		free(metrics_entry->bcn_rsp);
		metrics_entry->bcn_rsp = NULL;
	}
}

/*
 *When a process terminates, all of its memory is returned to the system,
 *including heap memory allocated by functions in the malloc package.
 *In programs that allocate memory and continue using it until program termination,
 *it is common to omit calls to free(), relying on this behavior to automatically free the memory.
 *This can be especially useful in programs that allocate many blocks of memory,
 *since adding multiple calls to free() could be expensive in terms of CPU time,
 *as well as perhaps being complicated to code
*/
void p1905_database_deinit(struct p1905_managerd_ctx *ctx)
{
	clear_topology_db(&ctx->topology_entry);
	/*free neighbor dev info*/
	clear_topology_neighbor_db(ctx->p1905_neighbor_dev, ctx->itf_number);
	clear_topology_device_db(&ctx->topodev_entry);
	clear_ap_capability_db(&ctx->ap_cap_entry);
	/*free map cpnfiguration info*/
	delete_exist_steering_policy(&ctx->map_policy.spolicy);
	delete_exist_metrics_policy(ctx, &ctx->map_policy.mpolicy);
	/*free metric info*/
	clear_metric_db(&ctx->metric_entry);
	/*free buffer*/
	if (ctx->ch_set) {
		free(ctx->ch_set);
		ctx->ch_set = NULL;
	}
	if (ctx->ch_stat) {
		free(ctx->ch_stat);
		ctx->ch_stat = NULL;
	}
	if (ctx->ch_rep) {
		free(ctx->ch_rep);
		ctx->ch_rep = NULL;
	}
	if (ctx->cli_steer_req) {
		free(ctx->cli_steer_req);
		ctx->cli_steer_req = NULL;
	}
	if (ctx->steer_cntrl) {
		free(ctx->steer_cntrl);
		ctx->steer_cntrl = NULL;
	}

	/*free all the agents info*/
	free_all_the_agents_info(&ctx->agent_head);

	debug(DEBUG_OFF, "==>\n");
}

void p1905_interface_init(struct p1905_managerd_ctx *ctx)
{
	int i = 0, j = 0;
#ifdef SUPPORT_WIFI
	for(i = 0; i < ctx->itf_number; i++) {
		if(ctx->itf[i].media_type & IEEE802_11_GROUP) {
		    ctx->itf[i].vs_info_length = 10;
		    ctx->itf[i].vs_info = malloc(ctx->itf[i].vs_info_length);
		    memset(ctx->itf[i].vs_info, 0, ctx->itf[i].vs_info_length);
		    memcpy(ctx->itf[i].vs_info, ctx->itf[i].mac_addr, 6);
			if (ctx->itf[i].is_wifi_sta) {
				*(ctx->itf[i].vs_info + 6) = 0x40;    //non-ap station
				memset(ctx->itf[i].vs_info, 0, 6);
			}
		}
	}
#endif

    /*init local bridge capability */
	ctx->br_cap[0].interface_num = ctx->itf_number - 1; /*skip virtual interface*/
	ctx->br_cap[0].itf_mac_list=malloc(ETH_ALEN * ctx->br_cap[0].interface_num);
	for(i = 0, j = FIRST_VITUAL_ITF + 1; i < ctx->br_cap[0].interface_num; i++, j++)
		memcpy(ctx->br_cap[0].itf_mac_list + i * ETH_ALEN, ctx->itf[j].mac_addr, ETH_ALEN);

	/*init local non-p1905.1 neighbor device information*/
	for(i = 0; i < ctx->itf_number; i++)
	{
		memcpy(ctx->non_p1905_neighbor_dev[i].local_mac_addr, ctx->itf[i].mac_addr, ETH_ALEN);
	    LIST_INIT(&(ctx->non_p1905_neighbor_dev[i].non_p1905nbr_head));
	}

	/*init local p1905.1 neighbor device information*/
	for(i = 0; i < ctx->itf_number; i++)
	{
		memcpy(ctx->p1905_neighbor_dev[i].local_mac_addr, ctx->itf[i].mac_addr, ETH_ALEN);
		LIST_INIT(&(ctx->p1905_neighbor_dev[i].p1905nbr_head));
	}
}

void p1905_interface_uninit(struct p1905_managerd_ctx *ctx)
{
    int i = 0;

    for(i=0;i<ITF_NUM;i++)
    {
        if(ctx->itf[i].vs_info_length > 0)
            free(ctx->itf[i].vs_info);
    }

    for(i=0;i<BRIDGE_NUM;i++)
    {
        free(ctx->br_cap[i].itf_mac_list);
    }
}

extern unsigned char dev_send_1905_buf[3072];
void discovery_at_boot_up(void *eloop_data, void *user_ctx);
void update_ethenet_port_status_at_bootup(void *eloop_data, void *user_ctx);

int p1905_managerd_init(struct p1905_managerd_ctx *ctx)
{
	char buf[32];
	struct ethernet_port_notifier noti;

    assert(ctx != NULL);

	ctx->rx_buf = (unsigned char*)os_malloc(MAX_PKT_LEN);
	if(NULL == ctx->rx_buf)
	{
	   debug(DEBUG_ERROR, "allocate rx buffer fail\n");
	   return -1;
	}

	ctx->revd_tlv = (unsigned char*)os_malloc(15360);
	if(NULL == ctx->revd_tlv)
	{
	   debug(DEBUG_ERROR, "allocate revd_tlv buffer fail\n");
	   return -1;
	}

    if(-1 == cmdu_init(ctx))
    {
    	debug(DEBUG_ERROR, "cmdu_init fail\n");
		os_free(ctx->rx_buf);
		ctx->rx_buf = NULL;
		os_free(ctx->revd_tlv);
		ctx->revd_tlv = NULL;
    	return -1;
    }
	if(-1 == concurrent_sock_init(ctx))
	{
		debug(DEBUG_ERROR, "concurrent_sock_init fail\n");
		os_free(ctx->rx_buf);
		ctx->rx_buf = NULL;
		os_free(ctx->revd_tlv);
		ctx->revd_tlv = NULL;
		return -1;
	}
#ifdef SUPPORT_CONTROL_SOCKET
	if(-1 == _1905_ctrl_sock_init(ctx))
	{
		debug(DEBUG_ERROR, "_1905_ctrl_sock_init fail\n");
		os_free(ctx->rx_buf);
		ctx->rx_buf = NULL;
		os_free(ctx->revd_tlv);
		ctx->revd_tlv = NULL;
		return -1;
	}
#endif
   	p1905_database_init(ctx);

	/*when certification, use wapp interface*/
	if(ctx->Certification)
	{
		ctx->wapp_ctrl = wapp_usr_intf_init(ctx);
		if(ctx->wapp_ctrl == NULL)
		{
			debug(DEBUG_ERROR, "wapp_usr_intf_init fail\n");
			os_free(ctx->rx_buf);
			ctx->rx_buf = NULL;
			os_free(ctx->revd_tlv);
			ctx->revd_tlv = NULL;
			return -1;
		}
		if(wapp_register_unsolicited_event(ctx, (unsigned char*)buf, 32, LIB_1905_CMDU_REQUEST) < 0)
		{
			debug(DEBUG_ERROR, "register LIB_1905_CMDU_REQUEST event fail\n");
			os_free(ctx->rx_buf);
			ctx->rx_buf = NULL;
			os_free(ctx->revd_tlv);
			ctx->revd_tlv = NULL;
			return -1;
		}
	}
	/*else, use api mode*/
	else
	{
		if(_1905_interface_init(ctx) < 0)
		{
			debug(DEBUG_ERROR, "_1905_interface_init fail\n");
			os_free(ctx->rx_buf);
			ctx->rx_buf = NULL;
			os_free(ctx->revd_tlv);
			ctx->revd_tlv = NULL;
			return -1;
		}
	}

	if ((ctx->sock_netlink = netlink_init(getpid())) < 0) {
		debug(DEBUG_ERROR, "netlink init fail\n");
	}

	if (mapfilter_netlink_init(getpid()) < 0) {
		debug(DEBUG_ERROR, "mapfilter_netlink_init init fail\n");
	}

    if(-1 == lldpdu_init(ctx))
    {
    	debug(DEBUG_ERROR, "lldpdu_init fail\n");
		os_free(ctx->rx_buf);
		ctx->rx_buf = NULL;
		os_free(ctx->revd_tlv);
		ctx->revd_tlv = NULL;
    	return -1;
    }

    if (ap_autoconfig_init(ctx)) {
		debug(DEBUG_ERROR, "fail to ap_autoconfig_init\n");
		return -1;
    }

	noti.down_handler = ethernet_pull_out_handler;
	noti.up_handler = ethernet_plug_in_handler;

	if (eth_layer_init((void *)ctx, &noti) < 0) {
		os_free(ctx->rx_buf);
		ctx->rx_buf = NULL;
		os_free(ctx->revd_tlv);
		ctx->revd_tlv = NULL;
		return -1;
	}

	/*init switch port status polling thread*/
	ctx->sock_internal = worker_task_receiver_sock_init("internal_server");
	if (ctx->sock_internal < 0) {
		debug(DEBUG_ERROR, "worker task receiver sock init fail\n");
		os_free(ctx->rx_buf);
		ctx->rx_buf = NULL;
		os_free(ctx->revd_tlv);
		ctx->revd_tlv = NULL;
		return -1;
	} else {
		if (worker_task_init() < 0) {
			debug(DEBUG_ERROR, "worker_task_init init fail\n");
			os_free(ctx->rx_buf);
			ctx->rx_buf = NULL;
			os_free(ctx->revd_tlv);
			ctx->revd_tlv = NULL;
			return -1;
		}
	}

	if (eloop_init()) {
		debug(DEBUG_ERROR, "Failed to initialize event loop");
		os_free(ctx->rx_buf);
		ctx->rx_buf = NULL;
		os_free(ctx->revd_tlv);
		ctx->revd_tlv = NULL;
		return -1;
	}

	eloop_register_signal_terminate(p1905_manage_terminate, NULL);

	/*set all socket to eloop*/
	eloop_register_read_sock(ctx->sock_br, cmdu_process, (void *)ctx, NULL);
	eloop_register_read_sock(ctx->sock_lldp, lldp_process, (void *)ctx, NULL);
	eloop_register_read_sock(ctx->_1905_ctrl.sock, library_cmd_process, (void *)ctx, NULL);
	eloop_register_read_sock(ctx->ctrl_sock, ctrl_cmd_process, (void *)ctx, NULL);
	if (ctx->sock_netlink > 0)
		eloop_register_read_sock(ctx->sock_netlink, netlink_event_process, (void *)ctx, NULL);
	eloop_register_read_sock(ctx->sock_internal, worker_task_event_process, (void *)ctx, NULL);
	if (ctx->Certification == 1)
		eloop_register_read_sock(ctx->wapp_ctrl->s, wapp_cmd_process, (void *)ctx, NULL);
	if (!ctx->MAP_Cer) {
		eloop_register_timeout(SWITCH_TABLE_DUMP_TIME, 0, non_1905_neighbor_update, (void *)ctx, NULL);
	}
	eloop_register_timeout(PERIODIC_TIMER_SEC, 0, p1905_managerd_periodic, (void *)ctx, NULL);
	eloop_register_timeout(PERIODIC_TIMER_SEC, 0, send_discovery_periodic, (void *)ctx, NULL);

//	eloop_register_timeout(10, 0, discovery_at_boot_up, (void *)ctx, NULL);
//	eloop_register_timeout(10, 0, update_ethenet_port_status_at_bootup, (void *)ctx, NULL);
    return 0;
}

#define RECV_FROM_BR      0x01
#define RECV_FROM_VIRTUAL 0x02

int p1905_managerd_run(struct p1905_managerd_ctx *ctx)
{
	eloop_run();
	return 0;
}

int common_process(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	/*if any cmdu need to be transmited , it will be sent here*/
	if (0 > process_cmdu_txq(ctx, buf)) {
	   debug(DEBUG_ERROR, "process_cmdu_txq fail\n");
	   return -1;
	}

	/*handle message wait queue*/
	handle_message_wait_queue(ctx, buf);

	return 0;
}

void cmdu_process(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_ctx;
	unsigned char *buf = ctx->rx_buf;
	struct ethhdr *eth_hdr = NULL;
	int len = MAX_PKT_LEN;
	int j = 0;
	unsigned char intf_mac[6];
	if (sock == ctx->sock_br) {
		len = cmdu_bridge_receive(ctx, ctx->sock_br, buf, MAX_PKT_LEN);
        if (0 > len) {
		   debug(DEBUG_ERROR, "cmdu receive failed on %s (%s)\n",ctx->br_name, strerror(errno));
           return;
        }
		eth_hdr = (struct ethhdr*)buf;
		if (get_receive_port_addr(ctx->br_name, eth_hdr->h_source, intf_mac) < 0) {
			debug(DEBUG_ERROR, "get_receive_port_addr fail, source=%02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(eth_hdr->h_source));
			return;
		}
		for (j = 0; j < ctx->itf_number; j++) {
			if (!os_memcmp(ctx->itf[j].mac_addr, intf_mac, sizeof(intf_mac))) {
				/*add this to tell which inf recv the cmdu*/
				ctx->recent_cmdu_rx_if_number = j;
				debug(DEBUG_TRACE, "recv cmdu from %s\n", ctx->itf[j].if_name);
				break;
			}
		}
		if (j >= ctx->itf_number) {
			debug(DEBUG_ERROR, "no any local interface match the mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(intf_mac));
			return;
		}
	}else if (sock == ctx->sock_inf_recv_1905[FIRST_VITUAL_ITF]) {
		len = cmdu_virtual_if_receive(ctx, sock, buf, MAX_PKT_LEN);
		if (0 > len) {
		   debug(DEBUG_ERROR, "cmdu_virtual_if_receive fail\n");
		   return;
		}
		ctx->recent_cmdu_rx_if_number = FIRST_VITUAL_ITF;
	}
	if (0 > cmdu_parse(ctx, buf, len)) {
	   debug(DEBUG_ERROR, "parsing cmdu fail \n");
	   return ;
	}

	common_process(ctx, buf);
}

void lldp_process(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_ctx;
	unsigned char *buf = ctx->rx_buf;
	int length = 0;

	/*reveive LLDPDU, process here*/
    debug(DEBUG_TRACE, "receive lldp message\n");
    length = receive_802_1_bridge_discovery_msg(ctx, buf, MAX_PKT_LEN);
    if (0 > length) {
       debug(DEBUG_ERROR, "receive 802.1 bridge discovery fail\n");
       return;
    }

    if (0 > parse_802_1_bridge_discovery_msg(ctx, buf)) {
        debug(DEBUG_ERROR, "parse 802.1 bridge discovery fail\n");
        return;
    }

	common_process(ctx, buf);
}


void library_cmd_process(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_ctx;
	unsigned char *buf = ctx->rx_buf;

	_1905_interface_receive_process(sock, ctx, 0);

	common_process(ctx, buf);
}

void ctrl_cmd_process(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_ctx;
	unsigned char *buf = ctx->rx_buf;

	_1905_ctrl_interface_recv_and_parse(ctx, (char *)buf, MAX_PKT_LEN);

	common_process(ctx, buf);
}

/*used in old Certification mode, 1905damon will directly receive command from wapp*/
void wapp_cmd_process(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_ctx;
	unsigned char *buf = ctx->rx_buf;
	int length = 0;

	length = wapp_usr_intf_recv(ctx, (char *)buf, MAX_PKT_LEN);
	if (0 > length) {
       debug(DEBUG_ERROR, "wapp_usr_intf_recv fail\n");
       return;
    }
	if (map_event_handler(ctx, (char *)buf, length, 0)) {
		debug(DEBUG_ERROR, "map_event_handler handle event fail\n");
	}

	common_process(ctx, buf);

	return;
}

void netlink_event_process(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_ctx;
	unsigned char *buf = ctx->rx_buf;
	int length = 0;

	length = netlink_event_recv(ctx->sock_netlink, buf, MAX_PKT_LEN);
	if (0 > length) {
       debug(DEBUG_ERROR, "netlink event recv fail\n");
       return;
    }
	buf = get_netlink_data(buf, &length);
//	hex_dump_all("netlink data", buf, length);
	/*add netlink event handler here*/
	netlink_event_handler(eloop_ctx, (struct _1905_netlink_message *)buf, length);

	common_process(ctx, buf);
	return;
}

void worker_task_event_process(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_ctx;
	unsigned char *buf = ctx->rx_buf;
	int length = 0;

	length = worker_task_receiver_recv(ctx->sock_internal, buf, MAX_PKT_LEN);
	if (0 > length) {
       debug(DEBUG_ERROR, "worker task receiver event recv fail\n");
       return;
    }
	worker_task_receiver_event_handler(eloop_ctx, (struct worker_task_message *)buf, length);

	common_process(ctx, buf);
	return;
}

void update_ethenet_port_status_at_bootup(void *eloop_data, void *user_ctx)
{
	static int i = 0;

	debug(DEBUG_ERROR, "get status of switch port[%d]\n", i);
	if(update_switch_port_status_by_netlink(i) < 0) {
		debug(DEBUG_ERROR, "get status of switch port[%d] fail\n", i);
	}
	i++;
	if (i >= MAX_LAN_PORT_NUM)
		return;
	eloop_register_timeout(1, 0/*500000*/, update_ethenet_port_status_at_bootup, eloop_data, NULL);
	return;
}

void discovery_at_boot_up(void *eloop_data, void *user_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_data;
	static int i = DISC_RETRY_TIME;
	int j = 0;

	for (j = 1; j < ctx->itf_number; j++) {
		/*send multicast cmdu to all the ethenet interface */
		if (ctx->itf[j].media_type == IEEE802_3_GROUP) {
			debug(DEBUG_TRACE, "send topology discovery with vs by interface[%s]\n", ctx->itf[j].if_name);
			ctx->mid++;
			insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
				e_topology_discovery_with_vendor_ie, ctx->mid, ctx->itf[j].if_name, 0);
		}
	}
	i--;

	if (i == 0) {
		i = DISC_RETRY_TIME;
		return;
	}

	eloop_register_timeout(3, 0, discovery_at_boot_up, (void *)ctx, NULL);

	return;
}


void p1905_managerd_uninit(struct p1905_managerd_ctx *ctx)
{
    assert(ctx != NULL);

	eth_layer_fini();
#ifdef MAP_R2
	cmd_set_ts_policy_clear(ctx);
#endif
	/*deinit all socket from eloop*/
	eloop_unregister_read_sock(ctx->sock_br);
	eloop_unregister_read_sock(ctx->sock_lldp);
	eloop_unregister_read_sock(ctx->_1905_ctrl.sock);
	eloop_unregister_read_sock(ctx->ctrl_sock);
	eloop_unregister_read_sock(ctx->sock_netlink);
	eloop_unregister_read_sock(ctx->sock_internal);
	if (ctx->Certification == 1)
		eloop_unregister_read_sock(ctx->wapp_ctrl->s);

	eloop_cancel_timeout(ap_controller_search_step, (void *)ctx, NULL);
	eloop_cancel_timeout(ap_autoconfig_enrolle_step, (void *)ctx, NULL);
	eloop_cancel_timeout(metrics_report_timeout, (void *)ctx, NULL);
	eloop_cancel_timeout(p1905_managerd_periodic, (void *)ctx, NULL);

	eloop_destroy();

    p1905_interface_uninit(ctx);
    cmdu_uninit(ctx);
    lldpdu_uninit(ctx);
	p1905_database_deinit(ctx);
	/*deinit topology tree*/
	clear_topology_tree(ctx);
	if(ctx->Certification == 1)
		wapp_usr_intf_deinit(ctx);
	else
		_1905_interface_deinit(ctx);
	netlink_deinit(ctx->sock_netlink);
	worker_task_deinit();
	worker_task_receiver_sock_deinit(ctx->sock_internal);
	if(ctx->rx_buf != NULL)
	{
		os_free(ctx->rx_buf);
		ctx->rx_buf = NULL;
	}
	if(ctx->revd_tlv != NULL)
	{
		os_free(ctx->revd_tlv);
		ctx->revd_tlv = NULL;
	}
}

unsigned char BtoH(char ch)
{
    if (ch >= '0' && ch <= '9') return (ch - '0');        /* Handle numerals*/
    if (ch >= 'A' && ch <= 'F') return (ch - 'A' + 0xA);  /* Handle capitol hex digits*/
    if (ch >= 'a' && ch <= 'f') return (ch - 'a' + 0xA);  /* Handle small hex digits*/
    return(255);
}


void AtoH(char *src, char *dest, int destlen)
{
    char *srcptr;
    char *destTemp;

    srcptr = src;
    destTemp = dest;

    while(destlen--)
    {
        *destTemp = BtoH(*srcptr++) << 4;    /* Put 1st ascii byte in upper nibble.*/
        *destTemp += BtoH(*srcptr++);      /* Add 2nd ascii byte to above.*/
        destTemp++;
    }
}

char * map_config_get_line(char *s,
	int size, FILE *stream, int *line, char **_pos)
{
	char *pos, *end, *sstart;

    while (fgets(s, size, stream)) {
        (*line)++;
        s[size - 1] = '\0';
        pos = s;

        /* Skip white space from the beginning of line. */
        while (*pos == ' ' || *pos == '\t' || *pos == '\r')
            pos++;

        /* Skip comment lines and empty lines */
        if (*pos == '#' || *pos == '\n' || *pos == '\0')
            continue;

        /*
         * Remove # comments unless they are within a double quoted
         * string.
		 */
        sstart = os_strchr(pos, '"');
        if (sstart)
            sstart = os_strrchr(sstart + 1, '"');
        if (!sstart)
            sstart = pos;
        end = os_strchr(sstart, '#');
        if (end)
            *end-- = '\0';
        else
            end = pos + os_strlen(pos) - 1;

        /* Remove trailing white space. */
        while (end > pos &&
               (*end == '\n' || *end == ' ' || *end == '\t' ||
            *end == '\r'))
            *end-- = '\0';

        if (*pos == '\0')
            continue;

		if (_pos)
            *_pos = pos;
        return pos;
    }

    if (_pos)
        *_pos = NULL;
    return NULL;
}

int map_read_config_file(struct p1905_managerd_ctx *ctx)
{
	FILE *file;
	char buf[256], *pos, *token, *token1;
	char tmpbuf[256];
	int line = 0, i = 0, ret = 0;
	unsigned char intf_idx = 0;

	if (strlen(ctx->map_cfg_file) == 0)
		strncpy(ctx->map_cfg_file, MAP_CFG_FILE, MAX_FILE_PATH_LENGTH - 1);

	file = fopen(ctx->map_cfg_file, "r");

	if (!file) {
		debug(DEBUG_ERROR, "open MAP cfg file (%s) fail\n", ctx->map_cfg_file);
		return -1;
	}
	os_memset(buf, 0, 256);
	os_memset(tmpbuf, 0, 256);

	while (map_config_get_line(buf, sizeof(buf), file, &line, &pos)) {
		if(strlen(pos) < 256)
			os_strcpy(tmpbuf, pos);
		token = strtok(pos, "=");

		if (token != NULL) {
			if (os_strcmp(token, "map_controller_alid") == 0) {
				if (ctx->role == CONTROLLER) {
					token = strtok(NULL, "");
					i = 0;
					token1 = strtok(token, ":");

					while (token1 != NULL) {
						AtoH(token1, (char *)&ctx->p1905_al_mac_addr[i], 1);
						i++;
						if (i >= ETH_ALEN)
							break;
						token1 = strtok(NULL, ":");
					}
					debug(DEBUG_OFF, "map_controller_alid = %02x:%02x:%02x:%02x:%02x:%02x\n",
						PRINT_MAC(ctx->p1905_al_mac_addr));
				}
			} else if (os_strcmp(token, "map_agent_alid") == 0) {
				if (ctx->role == AGENT) {
					token = strtok(NULL, "");
					i = 0;
					token1 = strtok(token, ":");

					while (token1 != NULL) {
						AtoH(token1, (char *)&ctx->p1905_al_mac_addr[i], 1);
						i++;
						if (i >= ETH_ALEN)
							break;
						token1 = strtok(NULL, ":");
					}
					debug(DEBUG_OFF, "map_agent_alid = %02x:%02x:%02x:%02x:%02x:%02x\n",
						PRINT_MAC(ctx->p1905_al_mac_addr));
				}
			} else if (os_strcmp(token, "br_inf") == 0) {
				token = strtok(NULL, "");
				os_strncpy((char *)ctx->br_name, token, (IFNAMSIZ-1));
				debug(DEBUG_OFF, "br_inf = %s\n", ctx->br_name);
			} else if (os_strcmp(token, "lan") == 0) {
				intf_idx = ctx->itf_number;
				token = strtok(NULL, "");
				os_strncpy((char *)ctx->itf[intf_idx].if_name, token, (IFNAMSIZ-1));
				debug(DEBUG_OFF, "lan = %s\n", ctx->itf[intf_idx].if_name);
				ctx->itf[intf_idx].media_type = IEEE802_3_GROUP;
				ctx->itf[intf_idx].is_wifi_sta = 0;
				ctx->itf_number++;
			} else if(os_strcmp(token, "veth") == 0) {
				intf_idx = ctx->itf_number;
				token = strtok(NULL, "");
				os_strncpy((char *)ctx->itf[intf_idx].if_name, token, (IFNAMSIZ-1));
				debug(DEBUG_OFF, "veth = %s\n", ctx->itf[intf_idx].if_name);
				ctx->itf[intf_idx].media_type = IEEE802_3_GROUP;
				ctx->itf[intf_idx].is_wifi_sta = 0;
				ctx->itf[intf_idx].is_veth = 1;
				ctx->itf_number++;
			}
#ifdef MAP_R2
 			else if (os_strcmp(token, "map_ver") == 0) {
 				token = strtok(NULL, "");
				if (os_strcmp(token, "R1") == 0)
					ctx->map_version = DEV_TYPE_R1;
				else
					ctx->map_version = DEV_TYPE_R2;
			}
#endif
			else if (os_strcmp(token, "discovery_cnt") == 0) {
				token = strtok(NULL, "");
				ctx->discovery_cnt = atoi(token);
				debug(DEBUG_OFF, "discovery_cnt = %d\n", ctx->discovery_cnt);
			} else if (os_strcmp(token, "bss_config_priority") == 0) {
				token = strtok(NULL, "");
				if (token == NULL)
					continue;
				if (os_strlen(token) >= (MAX_RADIO_NUM * (IFNAMSIZ + 1) * MAX_BSS_NUM)) {
					debug(DEBUG_ERROR, "bss_config_priority string too long\n");
					continue;
				}
				os_strcpy(ctx->bss_priority_str, token);
				debug(DEBUG_OFF, "bss_config_priority: %s\n", token);
			}
		}
	}

	fclose(file);

	if (ctx->discovery_cnt <= 0 || ctx->discovery_cnt > 60) {
		ctx->discovery_cnt = TOPOLOGY_DISCOVERY_COUNT;
		debug(DEBUG_OFF, "set discovery_cnt to defaut value(%d)\n",
			ctx->discovery_cnt);
	}

	return ret;
}

int map_read_wifi_file(struct p1905_managerd_ctx *ctx)
{
	FILE *file;
	char buf[256], *pos, *token, *token1, *mac, *identifer;
	int line = 0, ret = 0;
	unsigned char intf_idx = 0;
	char band[5] = {0};
	char phymode[3] = {0};
	int i;

	file = fopen(MAP_WIFI_FILE, "r");

	if (!file) {
		debug(DEBUG_ERROR, "open MAP wifi file (%s) fail\n", MAP_WIFI_FILE);
		return -1;
	}

	os_memset(buf, 0, 256);

	while (map_config_get_line(buf, sizeof(buf), file, &line, &pos)) {
		token = strtok(pos, "=");
		if (token != NULL) {
			/*interface=ra0;wiap;2.4g;N */
			if (os_strcmp(token, "interface") == 0) {
				intf_idx = ctx->itf_number;
				token = strtok(NULL, ";");
				/*interface name*/
				if (token != NULL) {
					os_strncpy((char *)ctx->itf[intf_idx].if_name, token, (IFNAMSIZ-1));
					debug(DEBUG_OFF, "interface[%d] = %s\n", intf_idx, ctx->itf[intf_idx].if_name);
				} else {
					debug(DEBUG_ERROR, "no interface name\n");
					ret = -1;
					goto out;
				}

				token = strtok(NULL, ";");
				/*role type*/
				if (token != NULL) {
					if (os_strcmp(token, "wiap") == 0) {
						ctx->itf[intf_idx].is_wifi_ap = 1;
					} else if (os_strcmp(token, "wista") == 0) {
						ctx->itf[intf_idx].is_wifi_sta = 1;
					} else {
						debug(DEBUG_ERROR, "unknow role[%d] = %s\n", intf_idx, token);
						ret = -1;
						goto out;
					}
					debug(DEBUG_TRACE, "role[%d] = %s\n", intf_idx, token);
				} else {
					debug(DEBUG_ERROR, "no role type\n");
					ret = -1;
					goto out;
				}

				token = strtok(NULL, ";");
				/*band*/
				if (token != NULL) {
					os_strncpy(band, token, 4);
					debug(DEBUG_TRACE, "band[%d] = %s\n", intf_idx, token);
				} else {
					debug(DEBUG_ERROR, "no band\n");
					ret = -1;
					goto out;
				}

				token = strtok(NULL, ";");
				/*phymode*/
				if (token != NULL) {
					os_strncpy(phymode, token, 2);
					debug(DEBUG_TRACE, "phymode[%d] = %s\n", intf_idx, token);
				} else {
					debug(DEBUG_ERROR, "no phymode\n");
					ret = -1;
					goto out;
				}

				if (os_strcmp(band, "2.4g") == 0 && os_strcmp(phymode, "B") == 0) {
					ctx->itf[intf_idx].media_type = IEEE802_11B_24G_GROUP;
					ctx->itf[intf_idx].radio_band = 0x00;
				} else if (os_strcmp(band, "2.4g") == 0 && os_strcmp(phymode, "G") == 0) {
					ctx->itf[intf_idx].media_type = IEEE802_11G_24G_GROUP;
					ctx->itf[intf_idx].radio_band = 0x00;
				} else if (os_strcmp(band, "2.4g") == 0 && os_strcmp(phymode, "N") == 0) {
					ctx->itf[intf_idx].media_type = IEEE802_11N_24G_GROUP;
					ctx->itf[intf_idx].radio_band = 0x00;
				} else if (os_strcmp(band, "2.4g") == 0 && os_strcmp(phymode, "AX") == 0) {
					ctx->itf[intf_idx].media_type = IEEE802_11AX_GROUP;
					ctx->itf[intf_idx].radio_band = 0x00;
				} else if (os_strcmp(band, "5g") == 0 && os_strcmp(phymode, "A") == 0) {
					ctx->itf[intf_idx].media_type = IEEE802_11A_5G_GROUP;
					ctx->itf[intf_idx].radio_band = 0x01;
				} else if (os_strcmp(band, "5g") == 0 && os_strcmp(phymode, "N") == 0) {
					ctx->itf[intf_idx].media_type = IEEE802_11N_5G_GROUP;
					ctx->itf[intf_idx].radio_band = 0x01;
				} else if (os_strcmp(band, "5g") == 0 && os_strcmp(phymode, "AC") == 0) {
					ctx->itf[intf_idx].media_type = IEEE802_11AC_5G_GROUP;
					ctx->itf[intf_idx].radio_band = 0x01;
				} else if (os_strcmp(band, "5g") == 0 && os_strcmp(phymode, "AX") == 0) {
					ctx->itf[intf_idx].media_type = IEEE802_11AX_GROUP;
					ctx->itf[intf_idx].radio_band = 0x01;
				}else {
					debug(DEBUG_ERROR, "band & phymode mismatch\n");
					ret = -1;
					goto out;
				}
				debug(DEBUG_TRACE, "media_type[%d] = %d\n", intf_idx, ctx->itf[intf_idx].media_type);

				token = strtok(NULL, ";");
				mac = token;
				if (mac == NULL) {
					debug(DEBUG_ERROR, "no interface mac\n");
					ret = -1;
					goto out;
				}
				token = strtok(NULL, ";");
				identifer = token;
				if (identifer == NULL) {
					debug(DEBUG_ERROR, "no identifier\n");
					ret = -1;
					goto out;
				}

				/*interface mac*/
				if (mac != NULL) {
					i = 0;
					token1 = strtok(mac, ":");
					while (token1 != NULL) {
						AtoH(token1, (char *)&ctx->itf[intf_idx].mac_addr[i], 1);
						i++;
						if (i >= ETH_ALEN)
							break;
						token1 = strtok(NULL, ":");
					}
					debug(DEBUG_TRACE, "mac_addr = %02x:%02x:%02x:%02x:%02x:%02x\n",
						PRINT_MAC(ctx->itf[intf_idx].mac_addr));
				}

				/*radio identifier*/
				if (identifer != NULL) {
					i = 0;
					token1 = strtok(identifer, ":");
					while (token1 != NULL) {
						AtoH(token1, (char *)&ctx->itf[intf_idx].identifier[i], 1);
						i++;
						if (i >= ETH_ALEN)
							break;
						token1 = strtok(NULL, ":");
					}
					debug(DEBUG_TRACE, "identifier = %02x:%02x:%02x:%02x:%02x:%02x\n",
						PRINT_MAC(ctx->itf[intf_idx].identifier));
				}

				ctx->itf_number++;
			}
		}
	}

	debug(DEBUG_OFF, "itf_number = %d\n", ctx->itf_number);

	if (ctx->itf_number > ITF_NUM) {
		debug(DEBUG_ERROR, "itf_number exceeds maximum!\n");
		ret = -1;
	}

out:
	fclose(file);
	return ret;
}

int set_bss_config_priority_by_str(struct p1905_managerd_ctx *ctx, char *prio_str)
{
	char *token = NULL;
	unsigned char priority = 1, i = 0;

	token = strtok(prio_str, ";");

	while (token != NULL) {
		for (i = 0; i < ctx->itf_number; i++) {
			if (os_strcmp((char*)ctx->itf[i].if_name, token) == 0) {
				ctx->itf[i].config_priority = priority;
				break;
			}
		}
		if (i >= ctx->itf_number) {
			debug(DEBUG_ERROR, "bss intf(%s) not exist in local interface\n", token);
		}
		debug(DEBUG_OFF, "\tbss intf(%s) priority=%d\n", token, priority);
		token = strtok(NULL, ";");
		priority++;
	}

	return 0;
}

int get_wireless_interfaces(struct p1905_managerd_ctx *ctx)
{
	char *token;
	char *cpy_str = NULL, *rem_strng = NULL;
	int cpy_str_len = 0;

	cpy_str_len = os_strlen(ctx->bss_priority_str);
	cpy_str = os_zalloc(cpy_str_len + 1);
	if (!cpy_str) {
		debug(DEBUG_ERROR, "alloc cpy_str fail\n");
		return -1;
	}
	os_memcpy(cpy_str, ctx->bss_priority_str, cpy_str_len);
	rem_strng = cpy_str;
	if (!strlen(rem_strng)) {
		debug(DEBUG_ERROR, "no wireless interface\n");
		os_free(cpy_str);
		return -1;
	}

	/* Static Interface list */
	token = strtok_r(rem_strng, ";", &rem_strng);
	/* Read each device line */
	while(token) {
		os_strncpy((char *)ctx->itf[ctx->itf_number++].if_name, token, (IFNAMSIZ-1));
		debug(DEBUG_OFF, "interface[%d] = %s\n", (ctx->itf_number - 1), token);
		token = strtok_r(rem_strng, ";", &rem_strng);
	}

	os_free(cpy_str);
	return 0;
}

int init_global_var_by_role(struct p1905_managerd_ctx *ctx)
{
	unsigned char vmac_enrollee[ETH_ALEN] = {0x00, 0x00, 0x00, 0x11, 0x22, 0x33};
	unsigned char vmac_registrar[ETH_ALEN] = {0x00, 0x00, 0x00, 0x33, 0x22, 0x11};
	unsigned char i = 0;

	/*init virual interface*/
	strncpy((char *)ctx->itf[FIRST_VITUAL_ITF].if_name, "vir_inf", strlen("vir_inf"));
	ctx->itf[FIRST_VITUAL_ITF].media_type = IEEE802_3_GROUP;
	ctx->itf[FIRST_VITUAL_ITF].is_veth = 1;
	if (ctx->role == CONTROLLER)
		memcpy(ctx->itf[FIRST_VITUAL_ITF].mac_addr, vmac_registrar, ETH_ALEN);
	else
		memcpy(ctx->itf[FIRST_VITUAL_ITF].mac_addr, vmac_enrollee, ETH_ALEN);
	ctx->itf_number = 1;

	/*init config info*/
	if (map_read_config_file(ctx) != 0) {
		debug(DEBUG_ERROR, "map_read_config_file fail\n");
		return -1;
	}

	/*read wireless interface information*/
	if (get_wireless_interfaces(ctx) != 0) {
		debug(DEBUG_ERROR, "get_wireless_interfaces fail\n");
		return -1;
	}

	/*set the bss config priority*/
	set_bss_config_priority_by_str(ctx, ctx->bss_priority_str);

	for(i = 0; i < ctx->itf_number; i++)
		ctx->itf[i].trx_config = 0;

	return 0;
}

#ifdef SUPPORT_BACKTRACE
void _1905_sigsegv_handler(int sig, void *signal_ctx)
{
	void *array[20];
	size_t size;

	debug(DEBUG_OFF, "sig(%d)!!!!!!!!!!\n",sig);

	// get void*'s for all entries on the stack
	size = backtrace(array, 20);

	// print out all the frames to stderr
	fprintf(stderr, "Error: signal %d:\n", sig);
	backtrace_symbols_fd(array, size, STDERR_FILENO);
	exit(1);
}

void _1905_enable_backtrack()
{
	eloop_register_signal(SIGSEGV, _1905_sigsegv_handler, NULL);
}
#else
void _1905_enable_backtrack() { }
#endif

#ifdef SUPPORT_COREDUMP
void _1905_enable_core_dump()
{
	struct rlimit limit;
	debug(DEBUG_OFF, "enabling core dump");
	limit.rlim_cur = 1024 * 1024 * 500;
	limit.rlim_max = 1024 * 1024 * 500;
	setrlimit(RLIMIT_CORE, &limit);
}
#else
void _1905_enable_core_dump() { }
#endif


/**
 * Main function
 *
 * \param  argc  number of arguments.
 * \param  argv  arguments pointer.
 * \return  error code.
 */
int main(int argc, char *argv[])
{
	struct p1905_managerd_ctx ctx;
    int result = 0;
	int c = 0;

	memset(&ctx, 0, sizeof(struct p1905_managerd_ctx));
	 //Open syslog file
    openlog("p1905_managerd", LOG_PID, LOG_DAEMON);

	while((c = getopt(argc, argv, "d:i:r:c:CBGMf:F:")) != -1)
	{
		switch(c)
		{
			case 'd':
			{
				debug_level = optarg[0] - '0';
				debug(DEBUG_OFF, "debug level = %d\n", debug_level);
				break;
			}
			case 'c':
			{
				ctx.concurrent = 1;
				break;
			}
			case 'r':
			{
				ctx.role = optarg[0] - '0';
				if (ctx.role > 1) {
					debug(DEBUG_ERROR, "role must be 0(controller) or 1(agent) fail!!!!\n");
					return -1;
				} else {
					debug(DEBUG_OFF, "set 1905 role %s\n", (ctx.role == 0)?"controller":"agent");
					break;
				}
			}
			case 'C':
			{
				ctx.Certification = 1;
				debug(DEBUG_OFF, "certification mode\n");
				break;
			}
			case 'B':
			{
				ctx.backtrace = 1;
				debug(DEBUG_OFF, "use backtrace\n");
				break;
			}
			case 'G':
			{
				ctx.core_dump= 1;
				debug(DEBUG_OFF, "use gdb to parse core_dump\n");
				break;
			}
			case 'f':
			{
				if(strlen(optarg) < MAX_FILE_PATH_LENGTH)
					strcpy(ctx.map_cfg_file, optarg);
				debug(DEBUG_OFF, "map cfg file %s\n", ctx.map_cfg_file);
			}
				break;

			case 'F':
			{
				if(strlen(optarg) < MAX_FILE_PATH_LENGTH)
					strcpy(ctx.wts_bss_cfg_file, optarg);
				debug(DEBUG_OFF, "wts bss cfg file %s\n", ctx.wts_bss_cfg_file);
			}
				break;
			case 'M':
			{
				ctx.MAP_Cer = 1;
				debug(DEBUG_OFF, "MAP Certification mode\n");
			}
				break;
			default:
				break;
		}
	}


	if (ctx.core_dump) {
		_1905_enable_core_dump();
	} else if (ctx.backtrace) {
		_1905_enable_backtrack();
	}

    if (0 != init_global_var_by_role(&ctx)) {
        debug(DEBUG_ERROR, "init_global_var_by_role fail!\n");
        result = -1;
        goto error;
    }

    //Initialize manager daemon
    if(0 != p1905_managerd_init(&ctx))
    {
        debug(DEBUG_ERROR, "p1905_managerd_init fail!\n");
        result = -1;
        goto error;
    }

    debug(DEBUG_OFF, "p1905_Manager Daemon Running\n");

    //Start manager daemon core
    p1905_managerd_run(&ctx);
    debug(DEBUG_OFF, "p1905_Manager Daemon exiting\n");


    //Uninitialize p1905_manager daemon
    p1905_managerd_uninit(&ctx);
error:
    //close syslog file
    closelog();

    return result;
}
