/*
 * cleopatre/application/p1905_managerd/src/cmdu_message.c
 *
 * (C) Copyright 2013 MSsar Semiconductor, Inc.
 *
 */
#include <stdio.h>
#include<stdlib.h>
#include <string.h>
#include <net/if.h>
#include <syslog.h>
#include <assert.h>
#include <linux/if_ether.h>
#include <asm/byteorder.h>
#include <pthread.h>
#include "cmdu_message.h"
#include "multi_ap.h"
#include "byteorder.h"
#include "debug.h"
#include "_1905_lib_io.h"
#include "os.h"
#include "eloop.h"
#include "cmdu.h"

extern unsigned char p1905_multicast_address[];

unsigned char tlv_temp[15360]={0};

extern unsigned short append_vendor_specific_type_tlv(unsigned char *pkt,
    struct p1905_vs_info *vs_info);
extern unsigned short append_link_metrics_query_type_tlv(unsigned char *pkt,
   unsigned char *target_al_mac, unsigned char type);
extern unsigned short append_transmitter_link_metrics_tlv(unsigned char *pkt,
   unsigned char *local_al_mac, unsigned char *neighbor_mac,
   unsigned char *local_itf_mac, unsigned char *neighbor_itf_mac,
   struct p1905_interface *itf_list, struct p1905_neighbor *devlist,
   struct p1905_managerd_ctx *ctx);
extern unsigned short append_push_button_event_notification_tlv(unsigned char *pkt,
   struct p1905_interface *itf_list);
extern unsigned short append_push_button_join_notification_tlv(unsigned char *pkt,
	unsigned char *al_id, unsigned short mid,
	unsigned char *local_mac, unsigned char *new_device_mac);
extern unsigned short append_receiver_link_metrics_tlv(unsigned char *pkt,
    unsigned char *local_al_mac, unsigned char *neighbor_mac,
    unsigned char *local_itf_mac, unsigned char *neighbor_itf_mac,
    struct p1905_interface *itf_list, struct p1905_managerd_ctx *ctx);
extern unsigned short append_link_metrics_result_code_tlv(unsigned char *pkt);


unsigned char *get_tlv_buffer(void)
{
    return tlv_temp;
}


/**
 *  delete all non-1905.1 neighbor device.
 *
 * \param  non_p1905_dev  pointer of non_1905.1 database
 * \param  retun error code
 */
void delete_non_p1905_neighbor_dev_info(struct non_p1905_neighbor *non_p1905_dev)
{
    int i = 0;
    struct non_p1905_neighbor_info *dev_info, *dev_info_temp;

    for(i=0;i<ITF_NUM;i++)
    {
        if(!LIST_EMPTY(&(non_p1905_dev[i].non_p1905nbr_head)))
        {
            dev_info = LIST_FIRST(&(non_p1905_dev[i].non_p1905nbr_head));
            while(dev_info)
            {
                dev_info_temp = LIST_NEXT(dev_info, non_p1905nbr_entry);
                LIST_REMOVE(dev_info, non_p1905nbr_entry);
                free(dev_info);
                dev_info = dev_info_temp;
            }
        }
    }
}

/**
 *  create topology discovery message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \param  al_mac  pointer of local abstraction layer mac address.
 * \param  itf_mac  pointer of transmitting ibterface mac address.
 * \return length of total tlvs included in this message
 */
unsigned short create_topology_discovery_message(unsigned char *buf,
                            unsigned char *al_mac, unsigned char *itf_mac,
                            unsigned short vs_len, unsigned char *vs_info)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;

    /*fill into tlvs*/
    length = append_1905_al_mac_addr_type_tlv(tlv_temp_buf,al_mac);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_mac_addr_type_tlv(tlv_temp_buf,itf_mac);
    total_tlvs_length += length;
    tlv_temp_buf += length;

	/*add vendor specific info if necessary*/
	if (vs_len != 0 && vs_info != NULL) {
		os_memcpy(tlv_temp_buf, vs_info, vs_len);
		total_tlvs_length += vs_len;
    	tlv_temp_buf += vs_len;
	}

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(TOPOLOGY_DISCOVERY);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

/**
 *  create topology discovery message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \param  al_mac  pointer of local abstraction layer mac address.
 * \param  itf_mac  pointer of transmitting ibterface mac address.
 * \return length of total tlvs included in this message
 */
unsigned short create_vendor_specific_topology_discovery_message(unsigned char *buf,
                            unsigned char *al_mac, unsigned char *itf_mac,
                            unsigned short vs_len, unsigned char *vs_info)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;

	/*add vendor specific info if necessary*/
	if (vs_len != 0 && vs_info != NULL) {
		os_memcpy(tlv_temp_buf, vs_info, vs_len);
		total_tlvs_length += vs_len;
    	tlv_temp_buf += vs_len;
	} else {
		return 0;
	}

    /*fill into tlvs*/
    length = append_1905_al_mac_addr_type_tlv(tlv_temp_buf,al_mac);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_mac_addr_type_tlv(tlv_temp_buf,itf_mac);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(VENDOR_SPECIFIC);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

/**
 *  create topology query message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \return length of total tlvs included in this message
 */
unsigned short create_topology_query_message(unsigned char *buf
#ifdef MAP_R2
, unsigned char map_version
#endif
)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;

#ifdef MAP_R2

    length = append_map_version_tlv(tlv_temp_buf, map_version);
    total_tlvs_length += length;
    tlv_temp_buf += length;
#endif // #ifdef MAP_R2

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(TOPOLOGY_QUERY);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

/**
 *  create topology response message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \param  al_mac  pointer of local abstraction layer mac address.
 * \param  itf_list  pointer of local interfaces information.
 * \param  br_cap_list  pointer of local internal bridge information.
 * \param  p1905_dev  pointer of 1905.1 neighbor devices information.
 * \param  non_p1905_dev poniter of non-1905.1 device information
 * \param  tpgr_list  list head of received topology response database
 * \return length of total tlvs included in this message
 */
unsigned short create_topology_response_message(unsigned char *buf,
    unsigned char *al_mac, struct p1905_managerd_ctx* ctx,
    struct bridge_capabiltiy *br_cap_list, struct p1905_neighbor *p1905_dev,
    struct non_p1905_neighbor *non_p1905_dev, struct list_head_tprdb *tpgr_head,
    unsigned char service, struct list_head_oper_bss *opbss_head,
    struct list_head_assoc_clients *asscli_head, unsigned int cnt)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;
    int i = 0;
    unsigned char seperate = 0;

    msg_hdr = (cmdu_message_header*)buf;

    length = append_device_info_type_tlv(tlv_temp_buf, al_mac, ctx);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_device_bridge_capability_type_tlv(tlv_temp_buf,
                                                      br_cap_list);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    for (i=0; i<ctx->itf_number; i++) {
       do {
	       length = append_p1905_neighbor_device_type_tlv(tlv_temp_buf, p1905_dev,
	                    i, &seperate);
	        total_tlvs_length += length;
	        tlv_temp_buf += length;
	       } while(seperate);
    }

    seperate = 0;
    /*append non-p1905.1 device tlv*/
    for(i=0; i<ctx->itf_number; i++)
    {
        do
        {
            length = append_non_p1905_neighbor_device_type_tlv(tlv_temp_buf,
                     non_p1905_dev,i, &seperate);
            total_tlvs_length += length;
            tlv_temp_buf += length;
        }while(seperate);
    }

	length = append_supported_service_tlv(tlv_temp_buf, service);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_operational_bss_tlv(tlv_temp_buf, opbss_head);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_associated_clients_tlv(tlv_temp_buf, asscli_head, cnt);
	total_tlvs_length += length;
	tlv_temp_buf += length;

#ifdef MAP_R2

	length = append_map_version_tlv(tlv_temp_buf, ctx->map_version);
	total_tlvs_length += length;
	tlv_temp_buf += length;
#endif // #ifdef MAP_R2

	if (ctx->Certification == 0) {
		reset_stored_tlv(ctx);
		store_revd_tlvs(ctx, tlv_temp, total_tlvs_length);
		os_get_time(&ctx->own_topo_rsp_update_time);
		_1905_notify_topology_rsp_event(ctx, ctx->p1905_al_mac_addr, ctx->p1905_al_mac_addr);
	}

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(TOPOLOGY_RESPONSE);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

/**
 *  create topology notification message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \param  al_mac  pointer of local abstraction layer mac address.
 * \return length of total tlvs included in this message
 */
unsigned short create_topology_notification_message(unsigned char *buf,
        unsigned char *al_mac, struct map_client_association_event *assoc_evt,
        unsigned char notifier, unsigned short vs_len, unsigned char *vs_info)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;

    length = append_1905_al_mac_addr_type_tlv(tlv_temp_buf,al_mac);
    total_tlvs_length += length;
    tlv_temp_buf += length;
	if (notifier) {
		length = append_client_assoc_event_tlv(tlv_temp_buf, assoc_evt);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}

	/*add vendor specific info if necessary*/
	if (vs_len != 0 && vs_info != NULL) {
		os_memcpy(tlv_temp_buf, vs_info, vs_len);
		total_tlvs_length += vs_len;
    	tlv_temp_buf += vs_len;
	}

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(TOPOLOGY_NOTIFICATION);
    /* topology notification is a relay multicast message,
     * so set relay_indicator
     */
    msg_hdr->relay_indicator = 0x1;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

/**
 *  create vendor specific message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \param  vs_info  vendor specific information.
 * \return length of total tlvs included in this message
 */
unsigned short create_vendor_specific_message(unsigned char *buf,
                        struct p1905_managerd_ctx *ctx)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;


	length = append_send_tlv(tlv_temp_buf, ctx);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;
    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(VENDOR_SPECIFIC);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

/**
 *  create link metric query message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \param  target_al_mac  0: all neighbors, otherwise, mac address of target.
 * \param  type  TX_METRICS_ONLY/RX_METRICS_ONLY/BOTH_TX_AND_RX_METRICS.
 * \return length of total tlvs included in this message
 */
unsigned short create_link_metrics_query_message(unsigned char *buf,
                        unsigned char *target_al_mac, unsigned char type)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;

    //fill into tlvs
    length = append_link_metrics_query_type_tlv(tlv_temp_buf, target_al_mac, type);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(LINK_METRICS_QUERY);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

/**
 *  create link metric response message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \param  target_al_mac  0: all neighbors, otherwise, mac address of target.
 * \param  type  TX_METRICS_ONLY/RX_METRICS_ONLY/BOTH_TX_AND_RX_METRICS.
 * \param  local_al_mac  local AL mac.
 * \param  itf_list  pointer of local interfaces information.
 * \param  tpd_head  list head of topology discovery database .
 * \return length of total tlvs included in this message
 */
unsigned short create_link_metrics_response_message(unsigned char *buf,
        unsigned char *target_al_mac, unsigned char type,
        unsigned char *local_al_mac, struct p1905_interface *itf_list,
        struct p1905_neighbor *devlist, struct list_head_tpddb *tpd_head,
        struct p1905_managerd_ctx *ctx)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;
    unsigned char query_all_neighbor[ETH_ALEN] = {0,0,0,0,0,0};
    struct topology_discovery_db *dcv_db = NULL;
    unsigned char valid_neighbor = 0;

    msg_hdr = (cmdu_message_header*)buf;

	if(ctx->Certification == 1)
	{
	    //fill into tlvs
	    if(!memcmp(target_al_mac, query_all_neighbor, ETH_ALEN))
	    {
	        /*query all neighbors, response all neighbor's link metric*/
	        LIST_FOREACH(dcv_db, tpd_head, tpddb_entry)
	        {
	        	/*skip the first virtual interface*/
        	if (!memcmp(dcv_db->receive_itf_mac, ctx->itf[FIRST_VITUAL_ITF].mac_addr, ETH_ALEN))
					continue;
	            if(type == TX_METRICS_ONLY)
	            {
                length = append_transmitter_link_metrics_tlv(tlv_temp_buf,
                         local_al_mac, dcv_db->al_mac, (unsigned char *)dcv_db->receive_itf_mac,
                         (unsigned char *)dcv_db->itf_mac, itf_list, devlist, ctx);
	                total_tlvs_length += length;
	                tlv_temp_buf += length;
	            }
	            else if(type == RX_METRICS_ONLY)
	            {
                length = append_receiver_link_metrics_tlv(tlv_temp_buf,
                         local_al_mac, dcv_db->al_mac, (unsigned char *)dcv_db->receive_itf_mac,
                         (unsigned char *)dcv_db->itf_mac, itf_list, ctx);
	                total_tlvs_length += length;
	                tlv_temp_buf += length;
	            }
	            else if(type == BOTH_TX_AND_RX_METRICS)
	            {
                length = append_transmitter_link_metrics_tlv(tlv_temp_buf,
                         local_al_mac, dcv_db->al_mac, (unsigned char *)dcv_db->receive_itf_mac,
                         (unsigned char *)dcv_db->itf_mac, itf_list, devlist, ctx);
	                total_tlvs_length += length;
	                tlv_temp_buf += length;

                length = append_receiver_link_metrics_tlv(tlv_temp_buf,
                     local_al_mac, dcv_db->al_mac, (unsigned char *)dcv_db->receive_itf_mac,
                     (unsigned char *)dcv_db->itf_mac, itf_list, ctx);
	                total_tlvs_length += length;
	                tlv_temp_buf += length;
	            }
	         }
	    } else {
	        LIST_FOREACH(dcv_db, tpd_head, tpddb_entry)
	        {
				/*skip the first virtual interface*/
        	if (!memcmp(dcv_db->receive_itf_mac, ctx->itf[FIRST_VITUAL_ITF].mac_addr, ETH_ALEN))
					continue;
	            if(!memcmp(target_al_mac, dcv_db->al_mac, ETH_ALEN))
	            {
	                valid_neighbor = 1;

	                if(type == TX_METRICS_ONLY)
	                {
                    length = append_transmitter_link_metrics_tlv(tlv_temp_buf,
                             local_al_mac, dcv_db->al_mac, (unsigned char *)dcv_db->receive_itf_mac,
                             (unsigned char *)dcv_db->itf_mac, itf_list, devlist, ctx);
                    total_tlvs_length += length;
                    tlv_temp_buf += length;
                }
                else if(type == RX_METRICS_ONLY)
                {
                    length = append_receiver_link_metrics_tlv(tlv_temp_buf,
                             local_al_mac, dcv_db->al_mac, (unsigned char *)dcv_db->receive_itf_mac,
                             (unsigned char *)dcv_db->itf_mac, itf_list, ctx);
                    total_tlvs_length += length;
                    tlv_temp_buf += length;
                }
                else if(type == BOTH_TX_AND_RX_METRICS)
                {
                    length = append_transmitter_link_metrics_tlv(tlv_temp_buf,
                             local_al_mac, dcv_db->al_mac, (unsigned char *)dcv_db->receive_itf_mac,
                             (unsigned char *)dcv_db->itf_mac, itf_list, devlist, ctx);
                    total_tlvs_length += length;
                    tlv_temp_buf += length;

                    length = append_receiver_link_metrics_tlv(tlv_temp_buf,
                         local_al_mac, dcv_db->al_mac, (unsigned char *)dcv_db->receive_itf_mac,
                         (unsigned char *)dcv_db->itf_mac, itf_list, ctx);
                    total_tlvs_length += length;
                    tlv_temp_buf += length;
                }
                break;
            }
        }

	        if(valid_neighbor != 1)
	        {
	            length = append_link_metrics_result_code_tlv(tlv_temp_buf);
	            total_tlvs_length += length;
	            tlv_temp_buf += length;
	        }
	    }
	}
	else
	{
		length = append_send_tlv(tlv_temp_buf, ctx);
		tlv_temp_buf += length;
		total_tlvs_length += length;
	}
    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(LINK_METRICS_RESPONSE);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

/**
 *  create push button event notification message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \param  plc_mac  mac address of PLC interface.
 * \param  al_mac  local AL mac address.
 * \return length of total tlvs included in this message
 */
unsigned short push_button_event_notification_message(unsigned char *buf,
                unsigned char *al_mac, struct p1905_interface *itf_list)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;

    //fill into tlvs
    length = append_1905_al_mac_addr_type_tlv(tlv_temp_buf, al_mac);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_push_button_event_notification_tlv(tlv_temp_buf, itf_list);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(P1905_PB_EVENT_NOTIFY);
    /*push button event notification is a multicast relay message*/
    msg_hdr->relay_indicator = 0x1;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

/**
 *  create push button join notification message.
 *
 * \param  buf  pointer to cmdu message header start position.
 * \param  plc_mac  mac address of PLC interface.
 * \param  al_mac  local AL mac address.
 * \param  pbc_p        instance of push_button_param
 * \return length of total tlvs included in this message
 */
unsigned short push_button_join_notification_message(unsigned char *buf,
                unsigned char *local_mac, unsigned char *al_mac,
                push_button_param *pbc_p, unsigned char is_plc,
                unsigned char role)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;

    //fill into tlvs
    length = append_1905_al_mac_addr_type_tlv(tlv_temp_buf, al_mac);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    if(is_plc) {
        length = append_push_button_join_notification_tlv(tlv_temp_buf,
            pbc_p->al_id, pbc_p->mid, local_mac, pbc_p->info.new_station);
    } else {
    	if (role == CONTROLLER)
			length = append_push_button_join_notification_tlv(tlv_temp_buf,
            	pbc_p->al_id, pbc_p->mid, local_mac, pbc_p->wifi_info.new_station);
    }
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(P1905_PB_JOIN_NOTIFY);
    /*push button join notification is a multicast relay message*/
    msg_hdr->relay_indicator = 0x1;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

unsigned short ap_autoconfiguration_search_message(
        unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;
	unsigned char service = ctx->service;
	unsigned char band = IEEE802_11_band_5GL;
	static unsigned int i = 0;
	unsigned char radio_index = 0;

    msg_hdr = (cmdu_message_header*)buf;

    length = append_1905_al_mac_addr_type_tlv(tlv_temp_buf, ctx->p1905_al_mac_addr);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_searched_role_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

	/*find controller state*/
	if (ctx->current_autoconfig_info.radio_index == -1) {
		if(ctx->send_tlv_len != 0)
		{
			band = ctx->send_tlv[0];
		}
		else
		{
			band = (i % 2 == 0) ? IEEE802_11_band_2P4G : IEEE802_11_band_5GL;
			i++;
			debug(DEBUG_TRACE, "search controller state band=%d\n", band);
		}
	} else {/*config state*/
		radio_index = (unsigned char)ctx->current_autoconfig_info.radio_index;
		if(ctx->send_tlv_len != 0)
		{
			band = ctx->send_tlv[0];
		}
		else
		{
			band = ctx->rinfo[radio_index].band;
		}
	}
    length = append_autoconfig_freq_band_tlv(tlv_temp_buf, band);
    total_tlvs_length += length;
    tlv_temp_buf += length;

	length = append_supported_service_tlv(tlv_temp_buf, service);
    total_tlvs_length += length;
    tlv_temp_buf += length;

	length = append_searched_service_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

#ifdef MAP_R2

	length = append_map_version_tlv(tlv_temp_buf, ctx->map_version);
	total_tlvs_length += length;
	tlv_temp_buf += length;
#endif // #ifdef MAP_R2

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(AP_AUTOCONFIG_SEARCH);
    /*AP autoconfiguration search message is a multicast relay message*/
    msg_hdr->relay_indicator = 0x1;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

unsigned char find_support_band(struct p1905_managerd_ctx *ctx, unsigned char band)
{
	unsigned char i = 0, defaut_band = 0;

	for (i = 0; i < ctx->bss_config_num; i++) {
		if ((band == 0 && ctx->bss_config[i].oper_class[0] == '8')
			|| (band == 1 && ctx->bss_config[i].oper_class[0] == '1')) {
			debug(DEBUG_ERROR, "%s controller support band %s\n",__func__, band == 0 ? "2.4g" : "5g");
			break;
		}
	}

	if (i < ctx->bss_config_num)
		return band;
	else
		return defaut_band;
}

unsigned short ap_autoconfiguration_response_message(
        unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;
	unsigned char band = 0;
	unsigned char service = 0;

    msg_hdr = (cmdu_message_header*)buf;

    length = append_supported_role_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

	band = ctx->peer_search_band;
    length = append_supported_freq_band_tlv(tlv_temp_buf, band);
    total_tlvs_length += length;
    tlv_temp_buf += length;

	length = append_supported_service_tlv(tlv_temp_buf, service);
    total_tlvs_length += length;
    tlv_temp_buf += length;

#ifdef MAP_R2

    length = append_map_version_tlv(tlv_temp_buf, ctx->map_version);
    total_tlvs_length += length;
    tlv_temp_buf += length;
#endif // #ifdef MAP_R2

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(AP_AUTOCONFIG_RESPONSE);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

unsigned short ap_autoconfiguration_renew_message(
        unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;
	unsigned char band = 0;

    msg_hdr = (cmdu_message_header*)buf;

    length = append_1905_al_mac_addr_type_tlv(tlv_temp_buf, ctx->p1905_al_mac_addr);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_supported_role_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;
	if(ctx->send_tlv_len != 0)
	{
		band = ctx->send_tlv[0];
		debug(DEBUG_TRACE, "insert renew message for band(%s)\n", band ? "5g" : "2g");
	}

    length = append_supported_freq_band_tlv(tlv_temp_buf, band);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(AP_AUTOCONFIG_RENEW);
    msg_hdr->relay_indicator = 0x1;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

#define WSC_TLV_LENGTH 1024

struct wsc_context {
	struct p1905_managerd_ctx *ctx;
	unsigned char *pkt;
	unsigned char wsc_type;
	WSC_CONFIG config_data;
	unsigned char wfa_vendor_extension;
	unsigned short out_len;
};

void *create_m2_thread_func(void *arg)
{
	struct wsc_context *wsc_ctx = (struct wsc_context *)arg;
	wsc_ctx->out_len = append_WSC_tlv(wsc_ctx->pkt, wsc_ctx->ctx, wsc_ctx->wsc_type,
		&wsc_ctx->config_data, wsc_ctx->wfa_vendor_extension);

	return NULL;
}

int create_all_m2s(struct p1905_managerd_ctx *ctx, unsigned char *buf,
	unsigned char num, unsigned short *total_len)
{
	unsigned char i = 0, j = 0;
	int ret = 0;
	pthread_t m2_thread_id[16];
	struct wsc_context wsc[16];
	unsigned short len = 0;
	unsigned char *m2_buf = NULL;
	WIFI_UTILS_STATUS status = wifi_utils_success;

	*total_len = 0;

	m2_buf = os_malloc(WSC_TLV_LENGTH * num);
	if (!m2_buf) {
		debug(DEBUG_ERROR, "failed to allocate m2 buffer on controller\n");
		goto fail;
	}

	/*before building m2, get the corresponding wsc config_data firstly*/
	for (i = 0; i < num; i++) {
		memset(&wsc[j].config_data, 0, sizeof(WSC_CONFIG));

	    status = get_wsc_config((void *)ctx, &wsc[j].config_data, &wsc[j].wfa_vendor_extension);
		if (status == wifi_utils_success) {
			j++;
		}
	}

	/*number of j represents that only j configuration data have been get successfully*/
	num = j;

	for (i = 0; i < num; i++) {
		wsc[i].ctx = ctx;
		wsc[i].pkt = m2_buf + i * WSC_TLV_LENGTH;
		wsc[i].out_len = 0;
		wsc[i].wsc_type = MESSAGE_TYPE_M2;
		ret = pthread_create(&m2_thread_id[i], NULL, create_m2_thread_func, &wsc[i]);
		if (ret < 0) {
			debug(DEBUG_ERROR, "worker task init fail\n");
		}
	}

	for (i = 0; i < num; i++) {
		if (pthread_join(m2_thread_id[i], NULL)) {
			debug(DEBUG_ERROR, "error join thread.\n");
		}
	}

	for (i = 0; i < num; i++) {
		if (wsc[i].out_len > WSC_TLV_LENGTH) {
			debug(DEBUG_ERROR, "error!!!length(%d) of one wsc message is larger than 1024 bytes.\n",
				wsc[i].out_len);
			len = 0;
			break;
		}
		os_memcpy(buf + len, wsc[i].pkt, wsc[i].out_len);
		len += wsc[i].out_len;
	}

	*total_len = len;

	os_free(m2_buf);
	return 1;
fail:
	return -1;

}

unsigned short ap_autoconfiguration_wsc_message(
        unsigned char *buf, struct p1905_managerd_ctx *ctx, unsigned char *al_mac, unsigned char wsc_type)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;
	struct radio_basic_capability_db *bcap = NULL;
	unsigned char radio_index = 0;
	unsigned char total_bss_config_num = 0;
	struct basic_cap_db *cap = NULL;
	unsigned char band_cap = 0;

    msg_hdr = (cmdu_message_header*)buf;

	if (ctx->role == AGENT && wsc_type == MESSAGE_TYPE_M1) {
		length = append_WSC_tlv(tlv_temp_buf, ctx, wsc_type, NULL, 0);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	} else if (ctx->role == CONTROLLER && wsc_type == MESSAGE_TYPE_M2) {
		total_bss_config_num = ctx->peer_bss_need_config < ctx->bss_config_num ?
			ctx->peer_bss_need_config : ctx->bss_config_num;
		/*send tear down on this radio*/
		if (total_bss_config_num == 0)
			total_bss_config_num = 1;
		create_all_m2s(ctx, tlv_temp_buf, total_bss_config_num, &length);

		total_tlvs_length += length;
		tlv_temp_buf += length;
	}


	if (ctx->role == AGENT) {
		debug(DEBUG_TRACE, "prepare to append_ap_radio_basic_capability_tlv\n");
		if (ctx->current_autoconfig_info.radio_index != -1) {
			radio_index = (unsigned char)ctx->current_autoconfig_info.radio_index;
			find_ap_radio_basic_capability_byidentifier(ctx,
				ctx->rinfo[radio_index].identifier, &bcap);
			if(bcap) {
				length = append_ap_radio_basic_capability_tlv(ctx, tlv_temp_buf, bcap);
				total_tlvs_length += length;
				tlv_temp_buf += length;
				SLIST_FOREACH(cap, &bcap->bcap_head, basic_cap_entry) {
					band_cap |= get_bandcap(ctx, cap->op_class, cap->non_operch_num,
					 				cap->non_operch_list);
				}
				switch (band_cap) {
				case BAND_2G_CAP:
					ctx->cur_conf_radio = RADIO_2G_CAP;
				break;
				case BAND_5GL_CAP:
					ctx->cur_conf_radio = RADIO_5GL_CAP;
				break;
				case BAND_5GH_CAP:
					ctx->cur_conf_radio = RADIO_5GH_CAP;
				break;
				case BAND_5G_CAP:
					ctx->cur_conf_radio = RADIO_5G_CAP;
				break;
				default:
				break;
				}
				debug(DEBUG_TRACE, "cur_conf_radio(%02x)\n", ctx->cur_conf_radio);
			} else {
				debug(DEBUG_ERROR, "invalid ap radio basic capability\n");
				debug(DEBUG_ERROR, "identifier=%02x:%02x:%02x:%02x:%02x:%02x\n",
					PRINT_MAC(ctx->rinfo[radio_index].identifier));
			}
		} else {
			debug(DEBUG_ERROR, "error radio_index == -1!!!\n");
		}
	} else {
		debug(DEBUG_TRACE, "prepare to append_ap_radio_identifier_tlv\n");
		length = append_ap_radio_identifier_tlv(tlv_temp_buf, ctx->identifier);
		total_tlvs_length += length;
		tlv_temp_buf += length;

#ifdef MAP_R2

		length = append_default_8021Q_tlv(tlv_temp_buf, al_mac,
			ctx->bss_config_num, ctx->bss_config);
		total_tlvs_length += length;
		tlv_temp_buf += length;

		length = append_traffic_separation_tlv(tlv_temp_buf, al_mac,
			ctx->bss_config_num, ctx->bss_config);
		total_tlvs_length += length;
		tlv_temp_buf += length;
#endif // #ifdef MAP_R2
	}

#ifdef MAP_R2
    length = append_map_version_tlv(tlv_temp_buf, ctx->map_version);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	if ((ctx->role == AGENT) && (wsc_type == MESSAGE_TYPE_M1)
		&& (ctx->map_version == DEV_TYPE_R2)) {
		struct ap_radio_advan_cap cap;
		memcpy(cap.radioid, ctx->identifier,ETH_ALEN);
		cap.flag = 0xc0;

		length = append_ap_radio_advan_tlv(tlv_temp_buf, &cap);
		total_tlvs_length += length;
		tlv_temp_buf += length;

		length = append_r2_cap_tlv(tlv_temp_buf, &(ctx->ap_cap_entry.ap_r2_cap));
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}
#endif // #ifdef MAP_R2

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(AP_AUTOCONFIG_WSC);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

void detect_neighbor_existence(struct p1905_managerd_ctx *ctx,
	unsigned char *neth_almac, unsigned char *sta_mac)
{
	struct topology_discovery_db *tpg_db = NULL;
	struct neighbor_list_db *ndb = NULL;
	int i = 0, j = 0;
	unsigned char remote_almac[ETH_ALEN] = {0};
	unsigned char zero_mac[ETH_ALEN] = {0};

	if (neth_almac) {
		os_memcpy(remote_almac, neth_almac, ETH_ALEN);
		debug(DEBUG_OFF, RED("neth_almac(%02x:%02x:%02x:%02x:%02x:%02x) in notification\n"),
			PRINT_MAC(neth_almac));
	} else if (sta_mac) {
		debug(DEBUG_OFF, RED("sta(%02x:%02x:%02x:%02x:%02x:%02x) in notification\n"),
			PRINT_MAC(sta_mac));
		find_neighbor_almac_by_intf_mac(ctx, sta_mac, remote_almac);
	} else {
		debug(DEBUG_OFF, "unhandle event\n");
		return;
	}

	if (!os_memcmp(remote_almac, zero_mac, ETH_ALEN))
		return;

	LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry) {
		if (os_memcmp(tpg_db->al_mac, remote_almac, ETH_ALEN))
			continue;
		SLIST_FOREACH(ndb, &ctx->query_neighbor_head, neighbor_entry) {
			if (!os_memcmp(tpg_db->al_mac, ndb->nalmac, ETH_ALEN) &&
				!os_memcmp(tpg_db->itf_mac, ndb->nitf_mac, ETH_ALEN))
				break;
		}
		if (ndb)
			continue;
		ndb = (struct neighbor_list_db *)malloc(sizeof(struct neighbor_list_db));
		if (ndb == NULL) {
			debug(DEBUG_ERROR, "allocate struct neighbor_list_db fail\n");
			return;
		}
	    os_memcpy(ndb->nalmac, tpg_db->al_mac, ETH_ALEN);
		os_memcpy(ndb->nitf_mac, tpg_db->itf_mac, ETH_ALEN);
	    SLIST_INSERT_HEAD(&ctx->query_neighbor_head, ndb, neighbor_entry);
		for(i = 0; i < ctx->itf_number; i++) {
			if (os_memcmp(tpg_db->receive_itf_mac, ctx->itf[i].mac_addr, ETH_ALEN))
				continue;
			for(j = 0; j < 3; j++) {
				ctx->mid++;
				insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
					e_topology_discovery_with_vendor_ie, ctx->mid, ctx->itf[i].if_name, 0);
			}
			debug(DEBUG_OFF, RED("send discovery_with_vendor_ie to dev"
				"(%02x:%02x:%02x:%02x:%02x:%02x) on %s"), PRINT_MAC(remote_almac),
				ctx->itf[i].if_name);
			eloop_register_timeout(DETECT_NEIGHBOR_TIME, 0, recv_neighbor_disc_timeout,
				(void *)ctx, (void *)ndb);
			break;
		}
	}
}

void recv_neighbor_disc_timeout(void *eloop_data, void *user_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_data;
	struct neighbor_list_db *ndb = (struct neighbor_list_db *)user_ctx;
	struct topology_discovery_db *tpg_db = NULL, *tpg_db_tmp = NULL;
    struct topology_response_db *tpgr_db = NULL;
	unsigned char del_rsp_cnt = 0;

	tpg_db = LIST_FIRST(&ctx->topology_entry.tpddb_head);
	while (tpg_db) {
		tpg_db_tmp = LIST_NEXT(tpg_db, tpddb_entry);
		if (!os_memcmp(tpg_db->al_mac, ndb->nalmac, ETH_ALEN) &&
			!os_memcmp(tpg_db->itf_mac, ndb->nitf_mac, ETH_ALEN)) {
			LIST_REMOVE(tpg_db, tpddb_entry);
			debug(DEBUG_OFF, RED("del neighbor(%02x:%02x:%02x:%02x:%02x:%02x)"
				" itf(%02x:%02x:%02x:%02x:%02x:%02x)\n"),
				PRINT_MAC(tpg_db->al_mac), PRINT_MAC(tpg_db->itf_mac));
			delete_p1905_neighbor_dev_info(ctx, tpg_db->al_mac,
				(unsigned char *)tpg_db->receive_itf_mac);
			if (find_discovery_by_almac(ctx, tpg_db->al_mac) == NULL) {
				debug(DEBUG_OFF, "delete topology rsp (%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(tpg_db->al_mac));
				/*topo C--ETH--A1--ETH--A2
				*reboot A1; during A1 get start, A2 will get C topology discovery by ETH
				*PON with MAP logic will run and drop A1 discovery
				*Add this logic to solve the issue mentioned above
				*/
				tpgr_db = find_topology_rsp_by_almac(ctx, tpg_db->al_mac);
				if (tpgr_db) {
					if (tpgr_db->support_service == 0 || tpgr_db->support_service == 2) {
						if (ctx->conn_port.is_set) {
							ctx->conn_port.is_set = 0;
							debug(DEBUG_OFF, RED("clear conn_port=%d\n"), ctx->conn_port.port_num);
						}
					}
					delete_exist_topology_response_database(ctx, tpg_db->al_mac);
					del_rsp_cnt++;
				}
			}
			os_free(tpg_db);
		}
		tpg_db = tpg_db_tmp;
	}

	report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
		ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
       	ctx->service, &ctx->ap_cap_entry.oper_bss_head,
        &ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);

	if (del_rsp_cnt)
		mark_valid_topo_rsp_node(ctx);

	SLIST_REMOVE(&ctx->query_neighbor_head, ndb, neighbor_list_db, neighbor_entry);
	os_free(ndb);
}

void check_neighbor_discovery(struct p1905_managerd_ctx *ctx,
	unsigned char *al_mac, unsigned char *itf_mac)
{
	struct neighbor_list_db *ndb = NULL;
	int ret = 0;

	SLIST_FOREACH(ndb, &ctx->query_neighbor_head, neighbor_entry) {
		if (!os_memcmp(ndb->nalmac, al_mac, ETH_ALEN) &&
			!os_memcmp(ndb->nitf_mac, itf_mac, ETH_ALEN)) {
			break;
		}
	}
	if (!ndb)
		return;

	ret= eloop_is_timeout_registered(recv_neighbor_disc_timeout, ctx, ndb);
	if (ret) {
		debug(DEBUG_OFF, RED("receive discovery for "
			"neighbor(%02x:%02x:%02x:%02x:%02x:%02x)\n"),
			PRINT_MAC(al_mac));
		eloop_cancel_timeout(recv_neighbor_disc_timeout, ctx, ndb);
		SLIST_REMOVE(&ctx->query_neighbor_head, ndb, neighbor_list_db, neighbor_entry);
		os_free(ndb);
	}
}

